# Helper module for rendering reports.
module ReportHelper
  
  # Render the report.
  # * <tt>report</tt>: The report instance
  # * <tt>html_options</tt>: HTML options
  def render_report(report, html_options={})
    case report
    when ActiveWarehouse::Report::TableReport
      render_table_report(report, html_options)
    when ActiveWarehouse::Report::ChartReport
      render_chart_report(report, html_options)
    else
      raise "Unsupported report type: #{report.class}"
    end
  end
  
  private
  def render_table_report(report, html_options)
    cube = report.cube
    fact_class = report.fact_class
    column_dimension_class = report.column_dimension_class
    row_dimension_class = report.row_dimension_class
    
    column_dimension = report.column_dimension_name
    row_dimension = report.row_dimension_name
    
    column_hierarchy = report.column_hierarchy
    column_hierarchy_constraints = report.column_constraints
    column_filters = report.column_filters
    
    row_hierarchy = report.row_hierarchy
    row_hierarchy_constraints = report.row_constraints
    row_filters = report.row_filters
    
    fact_attributes = report.fact_attributes
    
    conditions = report.conditions
    
    html_params = report.html_params
    
    cstage = (params[:cstage] || report.column_stage).to_i
    rstage = (params[:rstage] || report.row_stage).to_i
    
    col_param_prefix = report.column_param_prefix
    row_param_prefix = report.row_param_prefix
    
    link_cell = report.link_cell
    format = report.format
    
    # Get the dimension instances
    #col_dims = find_column_dimensions(column_dimension_class, column_hierarchy, cstage, col_param_prefix)
    #row_dims = find_row_dimensions(row_dimension_class, row_hierarchy, rstage, row_param_prefix)
    
    # Get the filters
    filters = {}
    params.each do |key, value|
      if key =~ /^#{col_param_prefix}_(.*)/
        filters["#{column_dimension}.#{$1}"] = value
      end
      if key =~ /^#{row_param_prefix}_(.*)/
        filters["#{row_dimension}.#{$1}"] = value
      end
    end
    
    # Query the cube
    query_result = cube.query(
      :column_dimension_name => column_dimension, 
      :column_hierarchy_name => column_hierarchy, 
      :row_dimension_name => row_dimension, 
      :row_hierarchy_name => row_hierarchy, 
      :conditions => conditions, 
      :cstage => cstage, 
      :rstage => rstage, 
      :filters => filters
    )
    
    table_attributes = {}
    table_attributes[:class] = html_options[:report_class] ||= 'report'
    
    col_hierarchy_length = column_dimension_class.hierarchy(column_hierarchy).length
    row_hierarchy_length = row_dimension_class.hierarchy(row_hierarchy).length
    col_group = column_dimension_class.hierarchy(column_hierarchy)[cstage]
    row_group = row_dimension_class.hierarchy(row_hierarchy)[rstage]
    
    #puts "col group: #{col_group.inspect}"
    
    drill_down_col_dim_values = find_column_dimension_values(column_dimension_class, column_hierarchy, cstage, col_param_prefix)
    drill_down_row_dim_values = find_row_dimension_values(row_dimension_class, row_hierarchy, rstage, row_param_prefix)
    
    col_dim_values = column_dimension_class.available_values(col_group).collect { |v| v.to_s }
    row_dim_values = row_dimension_class.available_values(row_group).collect { |v| v.to_s }
    #puts "col dim values: #{col_dim_values.inspect}"
    
    col_dim_values = col_dim_values.delete_if{ |v| !drill_down_col_dim_values.include?(v) }
    row_dim_values = row_dim_values.delete_if{ |v| !drill_down_row_dim_values.include?(v) }
   
    if column_filters[col_group] && column_filters[col_group].length > 0
      col_dim_values = col_dim_values.delete_if{ |v| !column_filters[col_group].include?(v) }
    end
    if row_filters[row_group] && row_filters[row_group].length > 0
      row_dim_values = row_dim_values.delete_if{ |v| !row_filters[row_group].include?(v) }
    end

    #puts "col dim values: #{col_dim_values.inspect}"
    
    # build the paths back to the top of the two dimension hierarchies
    col_path = []
    row_path = []
    0.upto(cstage) do |s| 
      p = params["#{col_param_prefix}_#{column_dimension_class.hierarchy(column_hierarchy)[s]}"]
      col_path << p unless p.nil?
    end
    0.upto(rstage) do |s| 
      p = params["#{row_param_prefix}_#{row_dimension_class.hierarchy(row_hierarchy)[s]}"]
      row_path << p unless p.nil?
    end
    #col_path << col_group.to_s
    #row_path << row_group.to_s
    
    # build the XHTML
    x = Builder::XmlMarkup.new(:indent => 2)
    x.table(table_attributes) do |x|
      x.tr do |x| # column dimension
        x.th
        col_dim_values.each do |col_dim_value|
          x.th({:colspan => fact_attributes.length}) do |x|
            x << link_to_if((cstage < col_hierarchy_length - 1), col_dim_value, {
              :cstage => cstage + 1, 
              :rstage => rstage, 
              "#{col_param_prefix}_#{col_group}" => col_dim_value}.merge(cube_params).merge(html_params))
          end
        end
      end
      
      x.tr do |x| # aggregated fact headers
        x.th
        col_dim_values.each do |col_dim_value|
          #fact_class.aggregate_fields.each do |field_name, options|
          #  x.th(options[:label].to_s || field_name.to_s.humanize)
          fact_attributes.each do |fact_attribute|
            case fact_attribute
            when Symbol, String
              fact_attribute = fact_class.field_for_name(fact_attribute.to_s.dup)
              raise "Field name #{field_name} not defined in the fact #{fact_class}" if fact_attribute.nil?
            end
            x.th(fact_attribute.label.humanize.titleize) # TODO replace with sortable column header
          end
        end
      end
      
      row_dim_values.each do |row_dim_value| # rows
        cell_row_path = row_path + [row_dim_value]
        x.tr do |x|
          x.td do |x| # row dimension label
            x << link_to_if((rstage < row_hierarchy_length - 1), row_dim_value, {
              :cstage => cstage, 
              :rstage => rstage + 1, 
              "#{row_param_prefix}_#{row_group}" => row_dim_value}.merge(cube_params).merge(html_params))
          end
          
          col_dim_values.each do |col_dim_value| # aggregated facts
            cell_col_path = col_path + [col_dim_value]
            fact_attributes.each_with_index do |fact_attribute, index|
              x.td do |x|
                # If a list of fact attributes to be displayed is defined then pull them
                # out of the fact class
                case fact_attribute
                when Symbol, String
                  field_name = fact_attribute.to_s.dup
                  fact_attribute = fact_class.field_for_name(fact_attribute)
                  if fact_attribute.nil?
                    raise "Field name #{field_name} not defined in the fact #{fact_class}"
                  end
                end
                
                # Get the value
                value = ''
                case fact_attribute
                when ActiveWarehouse::AggregateField
                  value = query_result.value(cell_row_path.last, cell_col_path.last, fact_attribute.label)
                when ActiveWarehouse::CalculatedField
                  value = fact_attribute.calculate(query_result.values(cell_row_path.last, cell_col_path.last))
                end
                
                # Handle custom formatting if a formatting block is defined
                block = format[fact_attribute.name.to_sym]
                if block
                  value = block.call(value)
                else
                  value = value.to_s
                end
                
                x << link_to_if(
                  (cstage < col_hierarchy_length - 1 && rstage < row_hierarchy_length - 1 && link_cell),
                  value, {
                    :cstage => cstage + 1, 
                    :rstage => rstage + 1, 
                    "#{col_param_prefix}_#{col_group}" => col_dim_value, 
                    "#{row_param_prefix}_#{row_group}" => row_dim_value
                  }.merge(cube_params).merge(html_params)
                )
              end
            end
          end
        end
      end
      
    end
  end
  
  # TODO: implement chart rendering
  def render_chart_report(report, html_options)
    "Not Yet Implemented"
  end
  
  protected
  def cube_params
    p = {}
    params.each { |key, value| p[key] = value if key =~ /^r_/ or key =~ /^c_/ }
    p
  end
  def find_column_dimensions(dimension_class, hierarchy, current_stage, param_prefix='c')
    find_dimensions(dimension_class, hierarchy, current_stage, param_prefix)
  end
  
  def find_column_dimension_values(dimension_class, hierarchy, current_stage, param_prefix='c')
    find_dimension_values(dimension_class, hierarchy, current_stage, param_prefix)
  end
  
  def find_row_dimensions(dimension_class, hierarchy, current_stage, param_prefix='r')
    find_dimensions(dimension_class, hierarchy, current_stage, param_prefix)
  end
  
  def find_row_dimension_values(dimension_class, hierarchy, current_stage, param_prefix='r')
    find_dimension_values(dimension_class, hierarchy, current_stage, param_prefix)
  end
  
  def find_dimensions(dimension_class, hierarchy, current_stage, param_prefix)
    # Construct the column find options
    find_options = {}
    group_by = []
    conditions_sql = []
    conditions_args = []
    current_stage.to_i.downto(0) do |stage|
      level = dimension_class.hierarchy(hierarchy)[stage]
      group_by << level
      unless stage == current_stage
        conditions_sql << "#{level} = ?"
        conditions_args << params["#{param_prefix}_#{level}"]
      end
    end
    find_options[:conditions] = nil
    if conditions_args.length > 0
      find_options[:conditions] = [conditions_sql.join(" and ")] + conditions_args
    end
    find_options[:group] = find_options[:order] = group_by.join(',')
    dimension_class.find(:all, find_options)
  end
  
  def find_dimension_values(dimension_class, hierarchy, current_stage, param_prefix)
    # Construct the column find options
    find_options = {}
    group_by = []
    conditions_sql = []
    conditions_args = []
    current_stage.to_i.downto(0) do |stage|
      level = dimension_class.hierarchy(hierarchy)[stage]
      group_by << level
      unless stage == current_stage
        param_name = "#{param_prefix}_#{level}"
        conditions_sql << "#{level} = '#{params[param_name]}'" # TODO protect against injection
      else
        find_options[:select] = level
      end
    end
    
    find_options[:conditions] = nil
    find_options[:conditions] = conditions_sql.join(" AND ") if conditions_sql.length > 0
    find_options[:group] = find_options[:order] = group_by.join(',')
    
    q = "SELECT #{find_options[:select]} FROM #{dimension_class.table_name}"
    q << " WHERE #{find_options[:conditions]}" if find_options[:conditions]
    q << " GROUP BY #{find_options[:group]}" if find_options[:group]
    q << " ORDER BY #{find_options[:order]}" if find_options[:order]
    
    puts "query: #{q}"

    dimension_class.connection.select_values(q)
  end
  
  public
  # Create a sortable column header.
  #
  # Parameters are
  #   column_name: The column display name
  #   order_by: The column name from the table used for sorting
  #   link_parameters: Parameters passed to the hypertext link
  #   attributes: Parameters passed to the <th> element
  #
  # Example:
  #   sortable_column_header('SSN','ssn')
  #   sortable_column_header('Name','name',{:onclick => 'doSomething();'}, {:title => 'Some Title'})
  def sortable_column_header(column_name, order_by, link_parameters={}, attributes={}) # TODO get this thing back into the mix
    order = 'desc'
    if params[:order_by].to_s == order_by.to_s
      sorted = 'sorted'
      if params[:order] == 'desc' || params[:order].nil?
        order_css_class = 'order_down'
        order = 'asc'
      elsif params[:order] == 'asc'
        order_css_class = 'order_up'
        order = 'desc'
      end
    else
      sorted = ''
    end
    
    attributes.merge!({:class => "sortable #{sorted} #{order_css_class}"})
    x.th(attributes) do |th|
      th << link_to(column_name, params.merge({:order_by => order_by, :order => order}), link_parameters)
    end
  end
end