require 'erb'

module ActiveWarehouse #:nodoc
  module Aggregate #:nodoc
    # A Pipelined implementation of a ROLAP engine that stores all possible
    # combinations
    # of fact and dimensional values for a specific cube.
    # 
    # This implementation attempts to reduce the amount of work required
    # by aggregating facts in a pipelined fashion.  This means that smaller
    # aggregates are generated from a preceding aggregate, in order to avoid
    # having to query the entire raw data set for every aggregate.
    # 
    # E.g.
    # 
    # ABCD -> ABC -> AB -> A -> *all*
    class PipelinedRolapAggregate < Aggregate
      include RolapCommon

      # Build and populate the data store
      def populate(options={})
        create_and_populate_all_table
        create_all_pipelined_agg_tables
        create_insert_statements.each_with_index do |insert, i|
          next if i == 0 #handled by create_and_populate_all_table
          connection.transaction {connection.execute(insert)}
        end
      end
      
      protected
      
      # build and populate a table which group by's all dimension columns.
      def create_and_populate_all_table
        dimension_column_names = dimensions_to_columns.collect do |c|
          "#{c.table_alias}.#{c.name} as #{c.table_alias}_#{c.name}"
        end
        
        fact_column_names = aggregate_fields.collect do |c|
          "#{c.from_table_name}.#{c.name} as #{c.label_for_table}"
        end
        
        sql = <<-SQL
          SELECT
          #{dimension_column_names.join(",")},
          #{aggregated_fact_column_sql_for_all}
          FROM #{tables_and_joins}
          GROUP BY
          #{dimensions_to_columns.collect{|c| "#{c.table_alias}.#{c.name}"}.join(",")}
        SQL
        
        all_table_name = indexed_rollup_table_name(dimension_column_names.length)
        
        sql = connection.add_select_into_table(all_table_name, sql)
        
        connection.drop_table(all_table_name) if connection.tables.include?(all_table_name)
        connection.transaction { connection.execute(sql) }
      end
      
      def create_all_pipelined_agg_tables
        (0..dimensions_to_columns.size-1).each do |i|
          create_rollup_cube_table(i)
        end
      end
      
      # Creates the rollup table
      def create_rollup_cube_table(index)
        table_name = indexed_rollup_table_name(index)
        connection.drop_table(table_name) if connection.tables.include?(table_name)
        
        ActiveRecord::Base.transaction do
          connection.create_table(table_name, :id => false) do |t|
            dimensions_to_columns.each do |c|
              t.column(c.label, c.column_type)
            end
            aggregate_fields.each do |c|
              options = {}
              options[:limit] = c.column_type == :integer ? 8 : c.limit
              options[:scale] = c.scale if c.scale
              options[:precision] = c.precision if c.precision
              t.column(c.label_for_table, c.column_type, options)
            end
          end
        end
      end
      
      def create_insert_statements
        dim_columns = dimensions_to_columns
        template_filename = File.dirname(__FILE__) + "/templates/pipelined_rollup_#{dim_columns.length}.sql"
        dim_columns.length.times do |i|
          eval("@dimension_#{i} = '#{dim_columns[i].label}'")
        end
        @aggregate_fields_from_flat_table = aggregated_fact_column_sql_for_rollup
        @aggregate_fields = aggregated_fact_column_sql_for_rollup
        @flat_table_name = flat_table_name
        @rollup_table_name = rollup_table_name
        
        inserts = []
        
        sql = ""
        ERB.new(File.read(template_filename)).result(binding).each do |line|
          if line.strip == ""
            inserts << sql
            sql = ""
          else
            sql += line
          end
        end
        
        inserts
      end
      
      def indexed_rollup_table_name(index)
        "#{rollup_table_name}_#{index}"
      end
      
      def aggregated_fact_column_sql_for_all
        aggregate_fields.collect { |c| 
          "#{c.strategy_name}(#{c.name}) AS #{c.label_for_table}"
        }.join(",")
      end
      
      def aggregated_fact_column_sql_for_rollup
        aggregate_fields.collect { |c|
          "#{c.strategy_name == :avg ? :avg : :sum}(#{c.label_for_table}) AS #{c.label_for_table}"
        }.join(", ")
      end
    end
  end
end