/*
 * Decompiled with CFR 0.152.
 */
package com.sleepycat.je.log;

import com.sleepycat.je.DatabaseException;
import com.sleepycat.je.EnvironmentStats;
import com.sleepycat.je.RunRecoveryException;
import com.sleepycat.je.StatsConfig;
import com.sleepycat.je.cleaner.TrackedFileSummary;
import com.sleepycat.je.cleaner.UtilizationTracker;
import com.sleepycat.je.config.EnvironmentParams;
import com.sleepycat.je.dbi.DbConfigManager;
import com.sleepycat.je.dbi.EnvironmentImpl;
import com.sleepycat.je.dbi.Operation;
import com.sleepycat.je.latch.Latch;
import com.sleepycat.je.latch.LatchSupport;
import com.sleepycat.je.log.ChecksumValidator;
import com.sleepycat.je.log.FileHandleSource;
import com.sleepycat.je.log.FileManager;
import com.sleepycat.je.log.FileSource;
import com.sleepycat.je.log.LogBuffer;
import com.sleepycat.je.log.LogBufferPool;
import com.sleepycat.je.log.LogEntryHeader;
import com.sleepycat.je.log.LogEntryType;
import com.sleepycat.je.log.LogException;
import com.sleepycat.je.log.LogFileNotFoundException;
import com.sleepycat.je.log.LogSource;
import com.sleepycat.je.log.entry.LogEntry;
import com.sleepycat.je.utilint.DbLsn;
import com.sleepycat.je.utilint.TestHook;
import com.sleepycat.je.utilint.Tracer;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.BufferOverflowException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.util.List;

public abstract class LogManager {
    private static final String DEBUG_NAME;
    protected LogBufferPool logBufferPool;
    protected Latch logWriteLatch;
    private boolean doChecksumOnRead;
    private FileManager fileManager;
    protected EnvironmentImpl envImpl;
    private boolean readOnly;
    private int readBufferSize;
    private long lastLsnAtRecovery = -1L;
    private int nRepeatFaultReads;
    private long nTempBufferWrites;
    private TestHook readHook;
    static final /* synthetic */ boolean $assertionsDisabled;

    public LogManager(EnvironmentImpl envImpl, boolean readOnly) throws DatabaseException {
        this.envImpl = envImpl;
        this.fileManager = envImpl.getFileManager();
        DbConfigManager configManager = envImpl.getConfigManager();
        this.readOnly = readOnly;
        this.logBufferPool = new LogBufferPool(this.fileManager, envImpl);
        this.doChecksumOnRead = configManager.getBoolean(EnvironmentParams.LOG_CHECKSUM_READ);
        this.logWriteLatch = LatchSupport.makeLatch(DEBUG_NAME, envImpl);
        this.readBufferSize = configManager.getInt(EnvironmentParams.LOG_FAULT_READ_SIZE);
    }

    public boolean getChecksumOnRead() {
        return this.doChecksumOnRead;
    }

    public long getLastLsnAtRecovery() {
        return this.lastLsnAtRecovery;
    }

    public void setLastLsnAtRecovery(long lastLsnAtRecovery) {
        this.lastLsnAtRecovery = lastLsnAtRecovery;
    }

    public void resetPool(DbConfigManager configManager) throws DatabaseException {
        this.logBufferPool.reset(configManager);
    }

    public long logForceFlush(LogEntry item, boolean fsyncRequired) throws DatabaseException {
        return this.log(item, false, true, fsyncRequired, false, false, -1L, 0);
    }

    public long logForceFlip(LogEntry item) throws DatabaseException {
        return this.log(item, false, true, false, true, false, -1L, 0);
    }

    public long log(LogEntry item) throws DatabaseException {
        return this.log(item, false, false, false, false, false, -1L, 0);
    }

    public long log(LogEntry item, boolean isProvisional, boolean backgroundIO, long oldNodeLsn, int oldNodeSize) throws DatabaseException {
        return this.log(item, isProvisional, false, false, false, backgroundIO, oldNodeLsn, oldNodeSize);
    }

    private long log(LogEntry item, boolean isProvisional, boolean flushRequired, boolean fsyncRequired, boolean forceNewLogFile, boolean backgroundIO, long oldNodeLsn, int oldNodeSize) throws DatabaseException {
        if (this.readOnly) {
            return -1L;
        }
        boolean marshallOutsideLatch = item.getLogType().marshallOutsideLatch();
        ByteBuffer marshalledBuffer = null;
        UtilizationTracker tracker = this.envImpl.getUtilizationTracker();
        LogResult logResult = null;
        boolean shouldReplicate = this.envImpl.isReplicated() && item.getLogType().isTypeReplicated();
        try {
            LogEntryHeader header = null;
            if (marshallOutsideLatch) {
                header = new LogEntryHeader(item, isProvisional, shouldReplicate);
                marshalledBuffer = this.marshallIntoBuffer(header, item, isProvisional, shouldReplicate);
            }
            logResult = this.logItem(header, item, isProvisional, flushRequired, forceNewLogFile, oldNodeLsn, oldNodeSize, marshallOutsideLatch, marshalledBuffer, tracker, shouldReplicate);
        }
        catch (BufferOverflowException e) {
            throw new RunRecoveryException(this.envImpl, (Throwable)e);
        }
        catch (IOException e) {
            throw new DatabaseException(Tracer.getStackTrace(e), e);
        }
        if (fsyncRequired) {
            this.fileManager.groupSync();
        }
        this.envImpl.getCheckpointer().wakeupAfterWrite();
        if (logResult.wakeupCleaner) {
            tracker.activateCleaner();
        }
        if (backgroundIO) {
            this.envImpl.updateBackgroundWrites(logResult.entrySize, this.logBufferPool.getLogBufferSize());
        }
        return logResult.currentLsn;
    }

    protected abstract LogResult logItem(LogEntryHeader var1, LogEntry var2, boolean var3, boolean var4, boolean var5, long var6, int var8, boolean var9, ByteBuffer var10, UtilizationTracker var11, boolean var12) throws IOException, DatabaseException;

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected LogResult logInternal(LogEntryHeader header, LogEntry item, boolean isProvisional, boolean flushRequired, boolean forceNewLogFile, long oldNodeLsn, int oldNodeSize, boolean marshallOutsideLatch, ByteBuffer marshalledBuffer, UtilizationTracker tracker, boolean shouldReplicate) throws IOException, DatabaseException {
        int entrySize;
        LogEntryType entryType = item.getLogType();
        if (oldNodeLsn != -1L) {
            tracker.countObsoleteNode(oldNodeLsn, entryType, oldNodeSize);
        }
        if (marshallOutsideLatch) {
            entrySize = marshalledBuffer.limit();
            if (!$assertionsDisabled && header == null) {
                throw new AssertionError();
            }
        } else {
            if (!$assertionsDisabled && header != null) {
                throw new AssertionError();
            }
            header = new LogEntryHeader(item, isProvisional, shouldReplicate);
            entrySize = header.getSize() + header.getItemSize();
        }
        if (forceNewLogFile) {
            this.fileManager.forceNewLogFile();
        }
        boolean flippedFile = this.fileManager.bumpLsn(entrySize);
        long currentLsn = -1L;
        boolean wakeupCleaner = false;
        boolean usedTemporaryBuffer = false;
        boolean success = false;
        try {
            currentLsn = this.fileManager.getLastUsedLsn();
            wakeupCleaner = tracker.countNewLogEntry(currentLsn, entryType, entrySize);
            if (item.countAsObsoleteWhenLogged()) {
                tracker.countObsoleteNodeInexact(currentLsn, entryType, entrySize);
            }
            if (!marshallOutsideLatch) {
                marshalledBuffer = this.marshallIntoBuffer(header, item, isProvisional, shouldReplicate);
            }
            if (entrySize != marshalledBuffer.limit()) {
                throw new DatabaseException("Logged item entrySize= " + entrySize + " but marshalledSize=" + marshalledBuffer.limit() + " type=" + entryType + " currentLsn=" + DbLsn.getNoFormatString(currentLsn));
            }
            LogBuffer useLogBuffer = this.logBufferPool.getWriteBuffer(entrySize, flippedFile);
            marshalledBuffer = header.addPostMarshallingInfo(this.envImpl, marshalledBuffer, this.fileManager.getPrevEntryOffset());
            useLogBuffer.latchForWrite();
            try {
                ByteBuffer useBuffer = useLogBuffer.getDataBuffer();
                if (useBuffer.capacity() - useBuffer.position() < entrySize) {
                    this.fileManager.writeLogBuffer(new LogBuffer(marshalledBuffer, currentLsn));
                    usedTemporaryBuffer = true;
                    if (!$assertionsDisabled && useBuffer.position() != 0) {
                        throw new AssertionError();
                    }
                    ++this.nTempBufferWrites;
                } else {
                    useBuffer.put(marshalledBuffer);
                }
            }
            finally {
                useLogBuffer.release();
            }
            if (shouldReplicate) {
                this.envImpl.getReplicator().replicateOperation(Operation.PLACEHOLDER, marshalledBuffer);
            }
            success = true;
        }
        finally {
            if (!success) {
                this.fileManager.restoreLastPosition();
            }
        }
        if (!usedTemporaryBuffer) {
            this.logBufferPool.writeCompleted(currentLsn, flushRequired);
        }
        item.postLogWork(currentLsn);
        return new LogResult(currentLsn, wakeupCleaner, entrySize);
    }

    private ByteBuffer marshallIntoBuffer(LogEntryHeader header, LogEntry item, boolean isProvisional, boolean shouldReplicate) throws DatabaseException {
        int entrySize = header.getSize() + header.getItemSize();
        ByteBuffer destBuffer = ByteBuffer.allocate(entrySize);
        header.writeToLog(destBuffer);
        item.writeEntry(header, destBuffer);
        item.setLastLoggedSize(entrySize);
        destBuffer.flip();
        return destBuffer;
    }

    ByteBuffer putIntoBuffer(LogEntry item, long prevLogEntryOffset) throws DatabaseException {
        LogEntryHeader header = new LogEntryHeader(item, false, false);
        ByteBuffer destBuffer = this.marshallIntoBuffer(header, item, false, false);
        return header.addPostMarshallingInfo(this.envImpl, destBuffer, 0L);
    }

    public LogEntry getLogEntry(long lsn) throws DatabaseException {
        this.envImpl.checkIfInvalid();
        LogSource logSource = this.getLogSource(lsn);
        return this.getLogEntryFromLogSource(lsn, logSource);
    }

    LogEntry getLogEntry(long lsn, RandomAccessFile file) throws DatabaseException {
        return this.getLogEntryFromLogSource(lsn, new FileSource(file, this.readBufferSize, this.fileManager));
    }

    private LogEntry getLogEntryFromLogSource(long lsn, LogSource logSource) throws DatabaseException {
        try {
            long fileOffset = DbLsn.getFileOffset(lsn);
            ByteBuffer entryBuffer = logSource.getBytes(fileOffset);
            if (!$assertionsDisabled && entryBuffer.limit() - entryBuffer.position() < 30) {
                throw new AssertionError();
            }
            LogEntryHeader header = new LogEntryHeader(this.envImpl, entryBuffer, false);
            header.readVariablePortion(entryBuffer);
            ChecksumValidator validator = null;
            if (this.doChecksumOnRead) {
                validator = new ChecksumValidator();
                int headerSizeMinusChecksum = header.getSizeMinusChecksum();
                int itemStart = entryBuffer.position();
                entryBuffer.position(itemStart - headerSizeMinusChecksum);
                validator.update(this.envImpl, entryBuffer, headerSizeMinusChecksum, false);
                entryBuffer.position(itemStart);
            }
            int itemSize = header.getItemSize();
            if (entryBuffer.remaining() < itemSize) {
                entryBuffer = logSource.getBytes(fileOffset + (long)header.getSize(), itemSize);
                ++this.nRepeatFaultReads;
            }
            if (this.doChecksumOnRead) {
                validator.update(this.envImpl, entryBuffer, itemSize, false);
                validator.validate(this.envImpl, header.getChecksum(), lsn);
            }
            if (!$assertionsDisabled && !LogEntryType.isValidType(header.getType())) {
                throw new AssertionError((Object)("Read non-valid log entry type: " + header.getType()));
            }
            LogEntry logEntry = LogEntryType.findType(header.getType(), header.getVersion()).getNewLogEntry();
            logEntry.readEntry(header, entryBuffer, true);
            logEntry.setLastLoggedSize(itemSize + header.getSize());
            if (this.readHook != null) {
                this.readHook.doIOHook();
            }
            LogEntry logEntry2 = logEntry;
            return logEntry2;
        }
        catch (DatabaseException e) {
            throw e;
        }
        catch (ClosedChannelException e) {
            throw new RunRecoveryException(this.envImpl, "Channel closed, may be due to thread interrupt", e);
        }
        catch (Exception e) {
            throw new DatabaseException(e);
        }
        finally {
            if (logSource != null) {
                logSource.release();
            }
        }
    }

    public Object get(long lsn) throws DatabaseException {
        LogEntry entry = this.getLogEntry(lsn);
        return entry.getMainItem();
    }

    public LogSource getLogSource(long lsn) throws DatabaseException {
        LogBuffer logBuffer = this.logBufferPool.getReadBuffer(lsn);
        if (logBuffer == null) {
            try {
                return new FileHandleSource(this.fileManager.getFileHandle(DbLsn.getFileNumber(lsn)), this.readBufferSize, this.fileManager);
            }
            catch (LogFileNotFoundException e) {
                throw new LogFileNotFoundException(DbLsn.getNoFormatString(lsn) + ' ' + e.getMessage());
            }
        }
        return logBuffer;
    }

    public void flush() throws DatabaseException {
        if (!this.readOnly) {
            this.flushInternal();
            this.fileManager.syncLogEnd();
        }
    }

    public void flushNoSync() throws DatabaseException {
        if (!this.readOnly) {
            this.flushInternal();
        }
    }

    protected abstract void flushInternal() throws LogException, DatabaseException;

    public void loadStats(StatsConfig config, EnvironmentStats stats) throws DatabaseException {
        stats.setNRepeatFaultReads(this.nRepeatFaultReads);
        stats.setNTempBufferWrites(this.nTempBufferWrites);
        if (config.getClear()) {
            this.nRepeatFaultReads = 0;
            this.nTempBufferWrites = 0L;
        }
        this.logBufferPool.loadStats(config, stats);
        this.fileManager.loadStats(config, stats);
        if (!config.getFast()) {
            this.loadEndOfLogStat(stats);
        }
    }

    public abstract TrackedFileSummary getUnflushableTrackedSummary(long var1) throws DatabaseException;

    protected TrackedFileSummary getUnflushableTrackedSummaryInternal(long file) throws DatabaseException {
        return this.envImpl.getUtilizationTracker().getUnflushableTrackedSummary(file);
    }

    public abstract void removeTrackedFile(TrackedFileSummary var1) throws DatabaseException;

    protected void removeTrackedFileInternal(TrackedFileSummary tfs) {
        tfs.reset();
    }

    public abstract void countObsoleteNode(long var1, LogEntryType var3, int var4) throws DatabaseException;

    protected void countObsoleteNodeInternal(UtilizationTracker tracker, long lsn, LogEntryType type, int size) throws DatabaseException {
        tracker.countObsoleteNode(lsn, type, size);
    }

    public abstract void countObsoleteNodes(TrackedFileSummary[] var1) throws DatabaseException;

    protected void countObsoleteNodesInternal(UtilizationTracker tracker, TrackedFileSummary[] summaries) throws DatabaseException {
        for (int i = 0; i < summaries.length; ++i) {
            TrackedFileSummary summary = summaries[i];
            tracker.addSummary(summary.getFileNumber(), summary);
        }
    }

    public abstract void countObsoleteINs(List var1) throws DatabaseException;

    protected void countObsoleteINsInternal(List lsnList) throws DatabaseException {
        UtilizationTracker tracker = this.envImpl.getUtilizationTracker();
        for (int i = 0; i < lsnList.size(); ++i) {
            Long offset = (Long)lsnList.get(i);
            tracker.countObsoleteNode(offset, LogEntryType.LOG_IN, 0);
        }
    }

    public abstract void loadEndOfLogStat(EnvironmentStats var1) throws DatabaseException;

    void loadEndOfLogStatInternal(EnvironmentStats stats) {
        stats.setEndOfLog(this.fileManager.getLastUsedLsn());
    }

    public void setReadHook(TestHook hook) {
        this.readHook = hook;
    }

    static {
        $assertionsDisabled = !LogManager.class.desiredAssertionStatus();
        DEBUG_NAME = LogManager.class.getName();
    }

    static class LogResult {
        long currentLsn;
        boolean wakeupCleaner;
        int entrySize;

        LogResult(long currentLsn, boolean wakeupCleaner, int entrySize) {
            this.currentLsn = currentLsn;
            this.wakeupCleaner = wakeupCleaner;
            this.entrySize = entrySize;
        }
    }
}

