/* 
 * ========================================================================
 * 
 * Copyright 2003 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * ========================================================================
 */
package org.apache.cactus.integration.ant.container.orion;

import java.io.File;
import java.io.IOException;

import org.apache.cactus.integration.ant.container.AbstractJavaContainer;
import org.apache.cactus.integration.ant.util.ResourceUtils;
import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.taskdefs.Java;
import org.apache.tools.ant.types.FileSet;
import org.apache.tools.ant.types.FilterChain;
import org.apache.tools.ant.types.Path;
import org.apache.tools.ant.util.FileUtils;

/**
 * Basic support for the Orin application server.
 * 
 * @version $Id: AbstractOrionContainer.java 239130 2005-01-29 15:49:18Z vmassol $
 */
public abstract class AbstractOrionContainer extends AbstractJavaContainer
{

    // Instance Variables ------------------------------------------------------

    /**
     * The Orion 1.x installation directory.
     */
    private File dir;

    /**
     * The port to which the container should be bound.
     */
    private int port = 8080;

    /**
     * The temporary directory from which the container will be started.
     */
    private File tmpDir;

    // Public Methods ----------------------------------------------------------

    /**
     * Sets the Orion installation directory.
     * 
     * @param theDir The directory to set
     */
    public final void setDir(File theDir)
    {
        this.dir = theDir;
    }

    /**
     * Sets the port to which the container should listen.
     * 
     * @param thePort The port to set
     */
    public final void setPort(int thePort)
    {
        this.port = thePort;
    }

    /**
     * Sets the temporary directory from which to start the container.
     * 
     * @param theTmpDir The directory to set
     */
    public final void setTmpDir(File theTmpDir)
    {
        this.tmpDir = theTmpDir;
    }

    // AbstractContainer Implementation ----------------------------------------

    /**
     * Returns the port to which the container should listen.
     * 
     * @return The port
     */
    public final int getPort()
    {
        return this.port;
    }

    /**
     * @see org.apache.cactus.integration.ant.container.Container#init
     */
    public final void init()
    {
        if (!this.dir.isDirectory())
        {
            throw new BuildException(this.dir + " is not a directory");
        }
    }

    /**
     * @see org.apache.cactus.integration.ant.container.Container#shutDown
     */
    public final void shutDown()
    {
        // invoke the main class
        Java java = createJavaForShutDown();
        Path classpath = java.createClasspath();
        FileSet fileSet = new FileSet();
        fileSet.setDir(this.dir);
        fileSet.createInclude().setName("*.jar");
        classpath.addFileset(fileSet);
        java.setClassname("com.evermind.client.orion.OrionConsoleAdmin");
        java.createArg().setValue("ormi://" + this.getServer() + ":23791/");
        java.createArg().setValue("admin");
        java.createArg().setValue("password");
        java.createArg().setValue("-shutdown");
        java.execute();
    }
    
    // Private Methods ---------------------------------------------------------

    /**
     * Invokes the command to start the Orion application server.
     */
    protected final void invokeServer()
    {
        // invoke the main class
        Java java = createJavaForStartUp();
        Path classpath = java.createClasspath();
        FileSet fileSet = new FileSet();
        fileSet.setDir(this.dir);
        fileSet.createInclude().setName("*.jar");
        classpath.addFileset(fileSet);
        addToolsJarToClasspath(classpath);
        java.setClassname("com.evermind.server.ApplicationServer");
        java.createArg().setValue("-config");
        java.createArg().setFile(new File(tmpDir, "conf/server.xml"));
        java.execute();
    }

    /**
     * Prepares a temporary installation of the container and deploys the 
     * web-application.
     * 
     * @param theResourcePrefix The prefix to use when looking up container
     *        resource in the JAR
     * @param theDirName The name of the temporary container installation
     *        directory
     * @throws IOException If an I/O error occurs
     */
    protected final void prepare(String theResourcePrefix, String theDirName)
        throws IOException
    {
        FileUtils fileUtils = FileUtils.newFileUtils();
        FilterChain filterChain = createFilterChain();

        this.tmpDir = setupTempDirectory(this.tmpDir, theDirName);
        cleanTempDirectory(this.tmpDir);

        // Copy configuration files into the temporary container directory

        File confDir = createDirectory(tmpDir, "conf");

        // Configuration files are not the same whether we deploy a 
        // WAR or an EAR
        String sharePath = RESOURCE_PATH + theResourcePrefix + "/share";
        String specificPath;
        if (getDeployableFile().isWar())
        {
            specificPath = RESOURCE_PATH + theResourcePrefix + "/war";
        }
        else
        {
            specificPath = RESOURCE_PATH + theResourcePrefix + "/ear";
        }

        ResourceUtils.copyResource(getProject(),
            specificPath + "/server.xml",
            new File(confDir, "server.xml"), filterChain);
        ResourceUtils.copyResource(getProject(), 
            specificPath + "/application.xml",
            new File(confDir, "application.xml"), filterChain);
        ResourceUtils.copyResource(getProject(), 
            specificPath + "/default-web-site.xml",
            new File(confDir, "default-web-site.xml"), filterChain);

        ResourceUtils.copyResource(getProject(),
            sharePath + "/global-web-application.xml",
            new File(confDir, "global-web-application.xml"), filterChain);
        ResourceUtils.copyResource(getProject(),
            sharePath + "/mime.types",
            new File(confDir, "mime.types"), filterChain);
        ResourceUtils.copyResource(getProject(),
            sharePath + "/principals.xml",
            new File(confDir, "principals.xml"), filterChain);
        ResourceUtils.copyResource(getProject(),
            sharePath + "/rmi.xml",
            new File(confDir, "rmi.xml"), filterChain);

        // Create default web app (required by Orion unfortunately...)
        File defaultWebAppDir = createDirectory(tmpDir, 
            "default-web-app/WEB-INF");
        ResourceUtils.copyResource(getProject(),
            sharePath + "/web.xml",
            new File(defaultWebAppDir, "web.xml"), filterChain);       
        
        // Orion need to have a /persistence directory created, otherwise it
        // throws an error
        createDirectory(tmpDir, "persistence");

        // Directory where modules to be deployed are located
        File appDir = createDirectory(tmpDir, "applications");

        // Deployment directory (i.e. where Orion expands modules)
        createDirectory(tmpDir, "application-deployments");

        // Orion log directory
        createDirectory(tmpDir, "log");
               
        fileUtils.copyFile(getDeployableFile().getFile(),
            new File(appDir, getDeployableFile().getFile().getName()), 
            null, true);
    }

}
