/* 
 * ========================================================================
 * 
 * Copyright 2001-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * 
 * ========================================================================
 */
package org.apache.cactus.server;

import java.io.InputStream;

import java.net.MalformedURLException;
import java.net.URL;

import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;

import javax.servlet.RequestDispatcher;
import javax.servlet.Servlet;
import javax.servlet.ServletContext;
import javax.servlet.ServletException;

/**
 * Abstract wrapper around <code>ServletContext</code>. This class provides
 * a common implementation of the wrapper for the different servlet API. In
 * addition to implementing the <code>ServletContext</code> interface it
 * provides additional features helpful for writing unit tests. More
 * specifically the <code>getRequestDispatcher()</code> method is overrided
 * to return an request dispatcher wrapper. In addition logs generated by
 * calls to the <code>log()</code> methods can be retrieved and asserted by
 * calling the <code>getLogs()</code> method.
 *
 * @version $Id: AbstractServletContextWrapper.java 239054 2004-10-24 01:30:23Z felipeal $
 */
public abstract class AbstractServletContextWrapper implements ServletContext
{
    /**
     * The original servlet context object
     */
    protected ServletContext originalContext;

    /**
     * List of parameters set using the <code>setInitParameter()</code> method.
     */
    protected Hashtable initParameters;
    
    /**
     * The logs resulting from calling the <code>log()</code> methods
     */
    private Vector logs = new Vector();

    // Constructors  -------------------------------------------------------

    /**
     * @param theOriginalContext the original servlet context object
     */
    public AbstractServletContextWrapper(ServletContext theOriginalContext)
    {
        this.originalContext = theOriginalContext;
        this.initParameters = new Hashtable();
    }

    // New methods ---------------------------------------------------------

    /**
     * @return the original unmodified config object
     * @since 1.6
     */
    public ServletContext getOriginalContext()
    {
        return this.originalContext;
    }
    
    /**
     * Sets a parameter as if it were set in the <code>web.xml</code> file
     * (using the &lt;context-param&gt; element).
     *
     * @param theName the parameter's name
     * @param theValue the parameter's value
     */
    public void setInitParameter(String theName, String theValue)
    {
        this.initParameters.put(theName, theValue);
    }
    
    /**
     * Returns all the text logs that have been generated using the
     * <code>log()</code> methods so that it is possible to easily assert the
     * content of the logs. This method does not return the exceptions or
     * throwable sent for logging; it only returns the messages.
     *
     * @return the logs as a vector of strings (each string contains the
     *         message that was sent for logging).
     */
    public Vector getLogs()
    {
        return this.logs;
    }
    
    // Overridden methods --------------------------------------------------

    /**
     * @see ServletContext#setAttribute(String, Object)
     */
    public void setAttribute(String theName, Object theAttribute)
    {
        this.originalContext.setAttribute(theName, theAttribute);
    }

    /**
     * @see ServletContext#removeAttribute(String)
     */
    public void removeAttribute(String theName)
    {
        this.originalContext.removeAttribute(theName);
    }

    /**
     * Intercept the log call and add the message to an internal vector of
     * log messages that can then later be retrieved and asserted by the
     * test case writer. Note that the throwable is not saved.
     *
     * @param theMessage a <code>String</code> that describes the error or
     *        exception
     * @param theCause the <code>Throwable</code> error or exception
     *
     * @see #getLogs()
     * @see ServletContext#log(String, Throwable)
     */
    public void log(String theMessage, Throwable theCause)
    {
        if (theMessage != null)
        {
            this.logs.addElement(theMessage);
        }

        this.originalContext.log(theMessage, theCause);
    }

    /**
     * Intercept the log call and add the message to an internal vector of
     * log messages that can then later be retrieved and asserted by the
     * test case writer. Note that the throwable is not saved.
     *
     * @param theMessage a <code>String</code> that describes the error or
     *        exception
     *
     * @see #getLogs()
     * @see ServletContext#log(String)
     */
    public void log(String theMessage)
    {
        if (theMessage != null)
        {
            this.logs.addElement(theMessage);
        }

        this.originalContext.log(theMessage);
    }

    /**
     * Intercept the log call and add the message to an internal vector of
     * log messages that can then later be retrieved and asserted by the
     * test case writer. Note that the throwable is not saved.
     *
     * @param theException the exception to log
     * @param theMessage a <code>String</code> that describes the error or
     *        exception
     *
     * @see #getLogs()
     * @see ServletContext#log(Exception, String)
     *
     * @deprecated As of Java Servlet API 2.1, use
     *             {@link #log(String message, Throwable throwable)} instead.
     *             This method was originally defined to write an exception's
     *             stack trace and an explanatory error message to the servlet
     *             log file.
     */
    public void log(Exception theException, String theMessage)
    {
        if (theMessage != null)
        {
            this.logs.addElement(theMessage);
        }

        this.originalContext.log(theException, theMessage);
    }

    /**
     * @see ServletContext#getServlets()
     */
    public Enumeration getServlets()
    {
        return this.originalContext.getServlets();
    }

    /**
     * @see ServletContext#getServletNames()
     */
    public Enumeration getServletNames()
    {
        return this.originalContext.getServletNames();
    }

    /**
     * @see ServletContext#getServlet(String)
     */
    public Servlet getServlet(String theName) throws ServletException
    {
        return this.originalContext.getServlet(theName);
    }

    /**
     * @see ServletContext#getServerInfo()
     */
    public String getServerInfo()
    {
        return this.originalContext.getServerInfo();
    }

    /**
     * @see ServletContext#getResourceAsStream(String)
     */
    public InputStream getResourceAsStream(String thePath)
    {
        return this.originalContext.getResourceAsStream(thePath);
    }

    /**
     * @see ServletContext#getResource(String)
     */
    public URL getResource(String thePath) throws MalformedURLException
    {
        return this.originalContext.getResource(thePath);
    }

    /**
     * @param thePath a string specifying the pathname to the resource
     * @return our request dispatcher wrapper
     * @see ServletContext#getRequestDispatcher(String)
     */
    public RequestDispatcher getRequestDispatcher(String thePath)
    {
        RequestDispatcher wrappedDispatcher = null;

        RequestDispatcher originalDispatcher = 
            this.originalContext.getRequestDispatcher(thePath);

        if (originalDispatcher != null)
        {
            wrappedDispatcher = 
                new RequestDispatcherWrapper(originalDispatcher);
        }

        return wrappedDispatcher;
    }

    /**
     * @param theName a string specifying the name of a servlet to wrap
     * @return our request dispatcher wrapper or null if the servlet cannot
     *         be found.
     * @see ServletContext#getNamedDispatcher(String)
     */
    public RequestDispatcher getNamedDispatcher(String theName)
    {
        RequestDispatcher wrappedDispatcher = null;

        RequestDispatcher originalDispatcher = 
            this.originalContext.getNamedDispatcher(theName);

        if (originalDispatcher != null)
        {
            wrappedDispatcher = 
                new RequestDispatcherWrapper(originalDispatcher);
        }

        return wrappedDispatcher;
    }

    /**
     * @see ServletContext#getRealPath(String)
     */
    public String getRealPath(String thePath)
    {
        return this.originalContext.getRealPath(thePath);
    }

    /**
     * @see ServletContext#getMinorVersion()
     */
    public int getMinorVersion()
    {
        return this.originalContext.getMinorVersion();
    }

    /**
     * @see ServletContext#getMimeType(String)
     */
    public String getMimeType(String theFilename)
    {
        return this.originalContext.getMimeType(theFilename);
    }

    /**
     * @see ServletContext#getMajorVersion()
     */
    public int getMajorVersion()
    {
        return this.originalContext.getMajorVersion();
    }

    /**
     * @return the union of the parameters defined in the Redirector
     *         <code>web.xml</code> file and the one set using the
     *         <code>setInitParameter()</code> method.
     */
    public Enumeration getInitParameterNames()
    {
        Vector names = new Vector();

        // Add parameters that were added using setInitParameter()
        Enumeration en = this.initParameters.keys();

        while (en.hasMoreElements())
        {
            String value = (String) en.nextElement();

            names.add(value);
        }

        // Add parameters from web.xml
        en = this.originalContext.getInitParameterNames();

        while (en.hasMoreElements())
        {
            String value = (String) en.nextElement();

            // Do not add parameters that have been overriden by calling
            // the setInitParameter() method.
            if (!names.contains(value))
            {
                names.add(value);
            }
        }

        return names.elements();
    }

    /**
     * @param theName the name of the parameter's value to return
     * @return the value of the parameter, looking for it first in the list of
     *         parameters set using the <code>setInitParameter()</code> method
     *         and then in those set in <code>web.xml</code>.
     */
    public String getInitParameter(String theName)
    {
        // Look first in the list of parameters set using the
        // setInitParameter() method.
        String value = (String) this.initParameters.get(theName);

        if (value == null)
        {
            value = this.originalContext.getInitParameter(theName);
        }

        return value;
    }
    
    /**
     * @param theUripath a String specifying the context path of another web
     *        application in the container
     * @return our servlet context wrapper
     * @see ServletContext#getContext(String)
     */
    public ServletContext getContext(String theUripath)
    {
        ServletContext context = new ServletContextWrapper(
            this.originalContext.getContext(theUripath));

        return context;
    }

    /**
     * @see ServletContext#getAttributeNames()
     */
    public Enumeration getAttributeNames()
    {
        return this.originalContext.getAttributeNames();
    }

    /**
     * @see ServletContext#getAttribute(String)
     */
    public Object getAttribute(String theName)
    {
        return this.originalContext.getAttribute(theName);
    }
}
