/* 
 * This file is part of the Echo Web Application Framework (hereinafter "Echo").
 * Copyright (C) 2002-2004 NextApp, Inc.
 *
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 */

/**
 * Adds a window to the client model's collection of handled windows.
 *
 * E_id: The id of the window.
 * E_window The window object.
 */
function E_ClientModel_E_addWindow(E_id, E_window) {
    this.E_windows.E_put(E_id, E_window);
}

/**
 * Changes the root window.   The root window is the
 * open window whose controller frame will be submitted whenever a
 * user action occurs that requires server interaction.
 * Changing the rooot window is necessary when the user closes the
 * root window.
 *
 * The new root window will be notified that it has been chosen
 * as the root window, and must then create a new ClientModel 
 * object, disposing of the current one.
 * The reason for this is that this ClientModel object will have been
 * created by a now-defunct (closed) window, and some browsers (IE)
 * will throw JavaScript exceptions as a result.
 */
function E_ClientModel_E_changeRootWindow() {
    var E_windowNames = this.E_windows.E_keySet();
    var E_windowCount = E_windowNames.E_size();
    var E_windowName;
    var E_window;
    var E_i = 0;
    var E_newRootWindow = this.E_rootWindow;
    
    // Find a new suitable root window.
    while (E_i < E_windowCount && E_newRootWindow == this.E_rootWindow) {
        E_windowName = E_windowNames.E_get(E_i);
        E_window = this.E_windows.E_get(E_windowName)
        E_newRootWindow = E_window;
        ++E_i;
    }
    
    if (E_newRootWindow == this.E_rootWindow) {
        // No new suitable new root window was found.  
        this.E_rootWindow = null;
    } else {
        // A suitable new root window was found.
        this.E_rootWindow = E_newRootWindow;
        
        // Notify the new window root window of its status as the root window.
        E_window.E_setAsRootWindow();
    }
}

/**
 * Returns the root window's controller.
 */
function E_ClientModel_E_getController() {
    if (this.E_rootWindow) {
        return this.E_rootWindow.frames.E_cf;
    } else {
        return null;
    }
}

/**
 * Returns the root window.
 */
function E_ClientModel_E_getRootWindow() {
    return this.E_rootWindow;
}

/**
 * Returns the window with the specified Id.
 */
function E_ClientModel_E_getWindow(E_id) {
    return this.E_windows.E_get(E_id);
}

/**
 * Returns the number of windows.
 */
function E_ClientModel_E_getWindowCount() {
    return this.E_windows.E_size();
}

/**
 * Returns the ids of all available windows.
 */
function E_ClientModel_E_getWindowIds() {
    return this.E_windows.E_keySet();
}

/**
 * Removes a window from the client model's collection of handled windows.
 * Does NOT close the window.
 *
 * E_id: The id of the window.
 */
function E_ClientModel_E_removeWindow(E_id) {
    this.E_windows.E_remove(E_id);
}

/**
 * Verifies that all windows in the internal collection of handled windows
 * are open and are referencing this client model.
 *
 * E_correctErrors: A boolean flag indicating whether to correct errors.  If 
 *                  this flag is true and a window is found to not reference
 *                  this client model, it will be made to reference it.
 *                   
 */
function E_ClientModel_E_verify(E_correctErrors) {
    var E_windowNames = this.E_windows.E_keySet();
    var E_windowCount = E_windowNames.E_size();
    var E_window;
    var E_i;
    var E_errorsFound = false;
    
    // Iterate through all available windows.
    for (E_i = 0; E_i < E_windowCount; ++E_i) {
        // Retrieve a window.
        E_window = this.E_windows.E_get(E_windowNames.E_get(E_i));

        if (E_window) {
            if (E_window.closed || !E_window.E_getClientModel) {
                E_errorsFound = true;
            } else if (E_window.E_getClientModel() != this) {
                E_errorsFound = true;
                if (E_correctErrors) {
                    // Set windows client model to this object
                    // if window's client model is not set to this
                    // object and caller desires such errors to be
                    // corrected.
                    E_window.E_setClientModel(this);
                }
            }
        } else {
            E_errorsFound = true;
        }
    }
    
    return E_errorsFound;
}

/**
 * Constructor.
 *
 * E_thisWindowId: The Id of the window creating the client model.
 */
function E_ClientModel(E_thisWindowId) {
    this.E_windows = new E_Map();
    
    this.E_addWindow = E_ClientModel_E_addWindow;
    this.E_changeRootWindow = E_ClientModel_E_changeRootWindow;
    this.E_getController = E_ClientModel_E_getController;
    this.E_getRootWindow = E_ClientModel_E_getRootWindow;
    this.E_getWindow = E_ClientModel_E_getWindow;
    this.E_getWindowCount = E_ClientModel_E_getWindowCount;
    this.E_getWindowIds = E_ClientModel_E_getWindowIds;
    this.E_removeWindow = E_ClientModel_E_removeWindow;
    this.E_verify = E_ClientModel_E_verify;

    this.E_addWindow(E_thisWindowId, window);
    this.E_rootWindow = window;
}
