/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.client;



import java.util.Collection;
import java.util.SortedSet;
import org.opends.server.admin.ConfigurationClient;
import org.opends.server.admin.IllegalPropertyValueException;
import org.opends.server.admin.ManagedObjectDefinition;
import org.opends.server.admin.PropertyIsReadOnlyException;
import org.opends.server.admin.std.meta.ReplicationDomainCfgDefn.IsolationPolicy;
import org.opends.server.admin.std.server.ReplicationDomainCfg;
import org.opends.server.types.DN;



/**
 * A client-side interface for reading and modifying Replication
 * Domain settings.
 * <p>
 * A Replication Domain comprises of several Directory Servers sharing
 * the same synchronized set of data.
 */
public interface ReplicationDomainCfgClient extends ConfigurationClient {

  /**
   * Get the configuration definition associated with this Replication Domain.
   *
   * @return Returns the configuration definition associated with this Replication Domain.
   */
  ManagedObjectDefinition<? extends ReplicationDomainCfgClient, ? extends ReplicationDomainCfg> definition();



  /**
   * Gets the "base-dn" property.
   * <p>
   * Specifies the base DN of the replicated data.
   *
   * @return Returns the value of the "base-dn" property.
   */
  DN getBaseDN();



  /**
   * Sets the "base-dn" property.
   * <p>
   * Specifies the base DN of the replicated data.
   * <p>
   * This property is read-only and can only be modified during
   * creation of a Replication Domain.
   *
   * @param value The value of the "base-dn" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   * @throws PropertyIsReadOnlyException
   *           If this Replication Domain is not being initialized.
   */
  void setBaseDN(DN value) throws IllegalPropertyValueException, PropertyIsReadOnlyException;



  /**
   * Gets the "heartbeat-interval" property.
   * <p>
   * Specifies the heart-beat interval that the Directory Server will
   * use when communicating with Replication Servers.
   * <p>
   * The Directory Server expects a regular heart-beat coming from the
   * Replication Server within the specified interval. If a heartbeat
   * is not received within the interval, the Directory Server closes
   * its connection and connects to another Replication Server.
   *
   * @return Returns the value of the "heartbeat-interval" property.
   */
  long getHeartbeatInterval();



  /**
   * Sets the "heartbeat-interval" property.
   * <p>
   * Specifies the heart-beat interval that the Directory Server will
   * use when communicating with Replication Servers.
   * <p>
   * The Directory Server expects a regular heart-beat coming from the
   * Replication Server within the specified interval. If a heartbeat
   * is not received within the interval, the Directory Server closes
   * its connection and connects to another Replication Server.
   *
   * @param value The value of the "heartbeat-interval" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   */
  void setHeartbeatInterval(Long value) throws IllegalPropertyValueException;



  /**
   * Gets the "isolation-policy" property.
   * <p>
   * Specifies the behavior of the Directory Server if a write
   * operation is attempted on the data within the Replication Domain
   * when none of the configured Replication Servers are available.
   *
   * @return Returns the value of the "isolation-policy" property.
   */
  IsolationPolicy getIsolationPolicy();



  /**
   * Sets the "isolation-policy" property.
   * <p>
   * Specifies the behavior of the Directory Server if a write
   * operation is attempted on the data within the Replication Domain
   * when none of the configured Replication Servers are available.
   *
   * @param value The value of the "isolation-policy" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   */
  void setIsolationPolicy(IsolationPolicy value) throws IllegalPropertyValueException;



  /**
   * Gets the "replication-server" property.
   * <p>
   * Specifies the addresses of the Replication Servers within the
   * Replication Domain to which the Directory Server should try to
   * connect at startup time.
   * <p>
   * Addresses must be specified using the syntax: hostname:port
   *
   * @return Returns the values of the "replication-server" property.
   */
  SortedSet<String> getReplicationServer();



  /**
   * Sets the "replication-server" property.
   * <p>
   * Specifies the addresses of the Replication Servers within the
   * Replication Domain to which the Directory Server should try to
   * connect at startup time.
   * <p>
   * Addresses must be specified using the syntax: hostname:port
   *
   * @param values The values of the "replication-server" property.
   * @throws IllegalPropertyValueException
   *           If one or more of the new values are invalid.
   */
  void setReplicationServer(Collection<String> values) throws IllegalPropertyValueException;



  /**
   * Gets the "server-id" property.
   * <p>
   * Specifies a unique identifier for the Directory Server within the
   * Replication Domain.
   * <p>
   * Each Directory Server within the same Replication Domain must
   * have a different server ID. A Directory Server which is a member
   * of multiple Replication Domains may use the same server ID for
   * each of its Replication Domain configurations.
   *
   * @return Returns the value of the "server-id" property.
   */
  Integer getServerId();



  /**
   * Sets the "server-id" property.
   * <p>
   * Specifies a unique identifier for the Directory Server within the
   * Replication Domain.
   * <p>
   * Each Directory Server within the same Replication Domain must
   * have a different server ID. A Directory Server which is a member
   * of multiple Replication Domains may use the same server ID for
   * each of its Replication Domain configurations.
   * <p>
   * This property is read-only and can only be modified during
   * creation of a Replication Domain.
   *
   * @param value The value of the "server-id" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   * @throws PropertyIsReadOnlyException
   *           If this Replication Domain is not being initialized.
   */
  void setServerId(int value) throws IllegalPropertyValueException, PropertyIsReadOnlyException;



  /**
   * Gets the "window-size" property.
   * <p>
   * Specifies the window size that the Directory Server will use when
   * communicating with Replication Servers.
   *
   * @return Returns the value of the "window-size" property.
   */
  int getWindowSize();



  /**
   * Sets the "window-size" property.
   * <p>
   * Specifies the window size that the Directory Server will use when
   * communicating with Replication Servers.
   *
   * @param value The value of the "window-size" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   */
  void setWindowSize(Integer value) throws IllegalPropertyValueException;

}
