/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.meta;



import org.opends.server.admin.BooleanPropertyDefinition;
import org.opends.server.admin.ClassPropertyDefinition;
import org.opends.server.admin.client.AuthorizationException;
import org.opends.server.admin.client.CommunicationException;
import org.opends.server.admin.client.ConcurrentModificationException;
import org.opends.server.admin.client.ManagedObject;
import org.opends.server.admin.client.MissingMandatoryPropertiesException;
import org.opends.server.admin.client.OperationRejectedException;
import org.opends.server.admin.ManagedObjectAlreadyExistsException;
import org.opends.server.admin.ManagedObjectDefinition;
import org.opends.server.admin.PropertyProvider;
import org.opends.server.admin.server.ConfigurationChangeListener;
import org.opends.server.admin.server.ServerManagedObject;
import org.opends.server.admin.std.client.EqualityMatchingRuleCfgClient;
import org.opends.server.admin.std.server.EqualityMatchingRuleCfg;
import org.opends.server.admin.std.server.MatchingRuleCfg;
import org.opends.server.admin.Tag;
import org.opends.server.types.DN;



/**
 * An interface for querying the Equality Matching Rule managed object
 * definition meta information.
 * <p>
 * Equality Matching Rules define a set of rules for performing
 * equality matching operations against assertion values (that is, to
 * determine whether an attribute value equals an assertion value).
 */
public final class EqualityMatchingRuleCfgDefn extends ManagedObjectDefinition<EqualityMatchingRuleCfgClient, EqualityMatchingRuleCfg> {

  // The singleton configuration definition instance.
  private static final EqualityMatchingRuleCfgDefn INSTANCE = new EqualityMatchingRuleCfgDefn();



  // Register the tags associated with this managed object definition.
  static {
    INSTANCE.registerTag(Tag.valueOf("core-server"));
  }



  /**
   * Get the Equality Matching Rule configuration definition
   * singleton.
   *
   * @return Returns the Equality Matching Rule configuration
   *         definition singleton.
   */
  public static EqualityMatchingRuleCfgDefn getInstance() {
    return INSTANCE;
  }



  /**
   * Private constructor.
   */
  private EqualityMatchingRuleCfgDefn() {
    super("equality-matching-rule", MatchingRuleCfgDefn.getInstance());
  }



  /**
   * {@inheritDoc}
   */
  public EqualityMatchingRuleCfgClient createClientConfiguration(
      ManagedObject<? extends EqualityMatchingRuleCfgClient> impl) {
    return new EqualityMatchingRuleCfgClientImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public EqualityMatchingRuleCfg createServerConfiguration(
      ServerManagedObject<? extends EqualityMatchingRuleCfg> impl) {
    return new EqualityMatchingRuleCfgServerImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public Class<EqualityMatchingRuleCfg> getServerConfigurationClass() {
    return EqualityMatchingRuleCfg.class;
  }



  /**
   * Get the "enabled" property definition.
   * <p>
   * Indicates whether the Equality Matching Rule is enabled for use.
   *
   * @return Returns the "enabled" property definition.
   */
  public BooleanPropertyDefinition getEnabledPropertyDefinition() {
    return MatchingRuleCfgDefn.getInstance().getEnabledPropertyDefinition();
  }



  /**
   * Get the "java-class" property definition.
   * <p>
   * Specifies the fully-qualified name of the Java class that
   * provides the Equality Matching Rule implementation.
   *
   * @return Returns the "java-class" property definition.
   */
  public ClassPropertyDefinition getJavaClassPropertyDefinition() {
    return MatchingRuleCfgDefn.getInstance().getJavaClassPropertyDefinition();
  }



  /**
   * Managed object client implementation.
   */
  private static class EqualityMatchingRuleCfgClientImpl implements
    EqualityMatchingRuleCfgClient {

    // Private implementation.
    private ManagedObject<? extends EqualityMatchingRuleCfgClient> impl;



    // Private constructor.
    private EqualityMatchingRuleCfgClientImpl(
        ManagedObject<? extends EqualityMatchingRuleCfgClient> impl) {
      this.impl = impl;
    }



    /**
     * {@inheritDoc}
     */
    public Boolean isEnabled() {
      return impl.getPropertyValue(INSTANCE.getEnabledPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setEnabled(boolean value) {
      impl.setPropertyValue(INSTANCE.getEnabledPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public String getJavaClass() {
      return impl.getPropertyValue(INSTANCE.getJavaClassPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setJavaClass(String value) {
      impl.setPropertyValue(INSTANCE.getJavaClassPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public ManagedObjectDefinition<? extends EqualityMatchingRuleCfgClient, ? extends EqualityMatchingRuleCfg> definition() {
      return INSTANCE;
    }



    /**
     * {@inheritDoc}
     */
    public PropertyProvider properties() {
      return impl;
    }



    /**
     * {@inheritDoc}
     */
    public void commit() throws ManagedObjectAlreadyExistsException,
        MissingMandatoryPropertiesException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException,
        CommunicationException {
      impl.commit();
    }

  }



  /**
   * Managed object server implementation.
   */
  private static class EqualityMatchingRuleCfgServerImpl implements
    EqualityMatchingRuleCfg {

    // Private implementation.
    private ServerManagedObject<? extends EqualityMatchingRuleCfg> impl;

    // The value of the "enabled" property.
    private final boolean pEnabled;

    // The value of the "java-class" property.
    private final String pJavaClass;



    // Private constructor.
    private EqualityMatchingRuleCfgServerImpl(ServerManagedObject<? extends EqualityMatchingRuleCfg> impl) {
      this.impl = impl;
      this.pEnabled = impl.getPropertyValue(INSTANCE.getEnabledPropertyDefinition());
      this.pJavaClass = impl.getPropertyValue(INSTANCE.getJavaClassPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void addEqualityChangeListener(
        ConfigurationChangeListener<EqualityMatchingRuleCfg> listener) {
      impl.registerChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeEqualityChangeListener(
        ConfigurationChangeListener<EqualityMatchingRuleCfg> listener) {
      impl.deregisterChangeListener(listener);
    }
    /**
     * {@inheritDoc}
     */
    public void addChangeListener(
        ConfigurationChangeListener<MatchingRuleCfg> listener) {
      impl.registerChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeChangeListener(
        ConfigurationChangeListener<MatchingRuleCfg> listener) {
      impl.deregisterChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public boolean isEnabled() {
      return pEnabled;
    }



    /**
     * {@inheritDoc}
     */
    public String getJavaClass() {
      return pJavaClass;
    }



    /**
     * {@inheritDoc}
     */
    public Class<? extends EqualityMatchingRuleCfg> configurationClass() {
      return EqualityMatchingRuleCfg.class;
    }



    /**
     * {@inheritDoc}
     */
    public DN dn() {
      return impl.getDN();
    }

  }
}
