/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.meta;



import java.util.Collection;
import java.util.SortedSet;
import java.util.TreeSet;
import org.opends.server.admin.AdministratorAction;
import org.opends.server.admin.AggregationPropertyDefinition;
import org.opends.server.admin.BooleanPropertyDefinition;
import org.opends.server.admin.client.AuthorizationException;
import org.opends.server.admin.client.CommunicationException;
import org.opends.server.admin.client.ConcurrentModificationException;
import org.opends.server.admin.client.ManagedObject;
import org.opends.server.admin.client.MissingMandatoryPropertiesException;
import org.opends.server.admin.client.OperationRejectedException;
import org.opends.server.admin.ManagedObjectAlreadyExistsException;
import org.opends.server.admin.ManagedObjectDefinition;
import org.opends.server.admin.PropertyIsReadOnlyException;
import org.opends.server.admin.PropertyOption;
import org.opends.server.admin.PropertyProvider;
import org.opends.server.admin.server.ConfigurationChangeListener;
import org.opends.server.admin.server.ServerManagedObject;
import org.opends.server.admin.std.client.NetworkGroupCfgClient;
import org.opends.server.admin.std.client.WorkflowCfgClient;
import org.opends.server.admin.std.server.NetworkGroupCfg;
import org.opends.server.admin.std.server.WorkflowCfg;
import org.opends.server.admin.StringPropertyDefinition;
import org.opends.server.admin.Tag;
import org.opends.server.admin.TopCfgDefn;
import org.opends.server.admin.UndefinedDefaultBehaviorProvider;
import org.opends.server.types.DN;



/**
 * An interface for querying the Network Group managed object
 * definition meta information.
 * <p>
 * The Network Group is used to classify incoming connections and
 * route requests to workflows.
 */
public final class NetworkGroupCfgDefn extends ManagedObjectDefinition<NetworkGroupCfgClient, NetworkGroupCfg> {

  // The singleton configuration definition instance.
  private static final NetworkGroupCfgDefn INSTANCE = new NetworkGroupCfgDefn();



  // The "enabled" property definition.
  private static final BooleanPropertyDefinition PD_ENABLED;



  // The "network-group-id" property definition.
  private static final StringPropertyDefinition PD_NETWORK_GROUP_ID;



  // The "workflow" property definition.
  private static final AggregationPropertyDefinition<WorkflowCfgClient, WorkflowCfg> PD_WORKFLOW;



  // Build the "enabled" property definition.
  static {
      BooleanPropertyDefinition.Builder builder = BooleanPropertyDefinition.createBuilder(INSTANCE, "enabled");
      builder.setOption(PropertyOption.MANDATORY);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "enabled"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<Boolean>());
      PD_ENABLED = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_ENABLED);
  }



  // Build the "network-group-id" property definition.
  static {
      StringPropertyDefinition.Builder builder = StringPropertyDefinition.createBuilder(INSTANCE, "network-group-id");
      builder.setOption(PropertyOption.READ_ONLY);
      builder.setOption(PropertyOption.MANDATORY);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "network-group-id"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<String>());
      PD_NETWORK_GROUP_ID = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_NETWORK_GROUP_ID);
  }



  // Build the "workflow" property definition.
  static {
      AggregationPropertyDefinition.Builder<WorkflowCfgClient, WorkflowCfg> builder = AggregationPropertyDefinition.createBuilder(INSTANCE, "workflow");
      builder.setOption(PropertyOption.MULTI_VALUED);
      builder.setOption(PropertyOption.READ_ONLY);
      builder.setOption(PropertyOption.MANDATORY);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "workflow"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<String>());
      builder.setParentPath("/");
      builder.setRelationDefinition("workflow");
      PD_WORKFLOW = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_WORKFLOW);
      INSTANCE.registerConstraint(PD_WORKFLOW.getSourceConstraint());
  }



  // Register the tags associated with this managed object definition.
  static {
    INSTANCE.registerTag(Tag.valueOf("core-server"));
  }



  /**
   * Get the Network Group configuration definition singleton.
   *
   * @return Returns the Network Group configuration definition
   *         singleton.
   */
  public static NetworkGroupCfgDefn getInstance() {
    return INSTANCE;
  }



  /**
   * Private constructor.
   */
  private NetworkGroupCfgDefn() {
    super("network-group", TopCfgDefn.getInstance());
  }



  /**
   * {@inheritDoc}
   */
  public NetworkGroupCfgClient createClientConfiguration(
      ManagedObject<? extends NetworkGroupCfgClient> impl) {
    return new NetworkGroupCfgClientImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public NetworkGroupCfg createServerConfiguration(
      ServerManagedObject<? extends NetworkGroupCfg> impl) {
    return new NetworkGroupCfgServerImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public Class<NetworkGroupCfg> getServerConfigurationClass() {
    return NetworkGroupCfg.class;
  }



  /**
   * Get the "enabled" property definition.
   * <p>
   * Indicates whether the Network Group is enabled for use in the
   * server.
   * <p>
   * If a network group is not enabled, its workflows will not be
   * accessible when processing operations.
   *
   * @return Returns the "enabled" property definition.
   */
  public BooleanPropertyDefinition getEnabledPropertyDefinition() {
    return PD_ENABLED;
  }



  /**
   * Get the "network-group-id" property definition.
   * <p>
   * Specifies the name that is used to identify the associated
   * Network Group .
   * <p>
   * The name must be unique among all the Network Groups in the
   * server.
   *
   * @return Returns the "network-group-id" property definition.
   */
  public StringPropertyDefinition getNetworkGroupIdPropertyDefinition() {
    return PD_NETWORK_GROUP_ID;
  }



  /**
   * Get the "workflow" property definition.
   * <p>
   * Identifies the workflows in the network group.
   *
   * @return Returns the "workflow" property definition.
   */
  public AggregationPropertyDefinition<WorkflowCfgClient, WorkflowCfg> getWorkflowPropertyDefinition() {
    return PD_WORKFLOW;
  }



  /**
   * Managed object client implementation.
   */
  private static class NetworkGroupCfgClientImpl implements
    NetworkGroupCfgClient {

    // Private implementation.
    private ManagedObject<? extends NetworkGroupCfgClient> impl;



    // Private constructor.
    private NetworkGroupCfgClientImpl(
        ManagedObject<? extends NetworkGroupCfgClient> impl) {
      this.impl = impl;
    }



    /**
     * {@inheritDoc}
     */
    public Boolean isEnabled() {
      return impl.getPropertyValue(INSTANCE.getEnabledPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setEnabled(boolean value) {
      impl.setPropertyValue(INSTANCE.getEnabledPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public String getNetworkGroupId() {
      return impl.getPropertyValue(INSTANCE.getNetworkGroupIdPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setNetworkGroupId(String value) throws PropertyIsReadOnlyException {
      impl.setPropertyValue(INSTANCE.getNetworkGroupIdPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public SortedSet<String> getWorkflow() {
      return impl.getPropertyValues(INSTANCE.getWorkflowPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setWorkflow(Collection<String> values) {
      impl.setPropertyValues(INSTANCE.getWorkflowPropertyDefinition(), values);
    }



    /**
     * {@inheritDoc}
     */
    public ManagedObjectDefinition<? extends NetworkGroupCfgClient, ? extends NetworkGroupCfg> definition() {
      return INSTANCE;
    }



    /**
     * {@inheritDoc}
     */
    public PropertyProvider properties() {
      return impl;
    }



    /**
     * {@inheritDoc}
     */
    public void commit() throws ManagedObjectAlreadyExistsException,
        MissingMandatoryPropertiesException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException,
        CommunicationException {
      impl.commit();
    }

  }



  /**
   * Managed object server implementation.
   */
  private static class NetworkGroupCfgServerImpl implements
    NetworkGroupCfg {

    // Private implementation.
    private ServerManagedObject<? extends NetworkGroupCfg> impl;

    // The value of the "enabled" property.
    private final boolean pEnabled;

    // The value of the "network-group-id" property.
    private final String pNetworkGroupId;

    // The value of the "workflow" property.
    private final SortedSet<String> pWorkflow;



    // Private constructor.
    private NetworkGroupCfgServerImpl(ServerManagedObject<? extends NetworkGroupCfg> impl) {
      this.impl = impl;
      this.pEnabled = impl.getPropertyValue(INSTANCE.getEnabledPropertyDefinition());
      this.pNetworkGroupId = impl.getPropertyValue(INSTANCE.getNetworkGroupIdPropertyDefinition());
      this.pWorkflow = impl.getPropertyValues(INSTANCE.getWorkflowPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void addChangeListener(
        ConfigurationChangeListener<NetworkGroupCfg> listener) {
      impl.registerChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeChangeListener(
        ConfigurationChangeListener<NetworkGroupCfg> listener) {
      impl.deregisterChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public boolean isEnabled() {
      return pEnabled;
    }



    /**
     * {@inheritDoc}
     */
    public String getNetworkGroupId() {
      return pNetworkGroupId;
    }



    /**
     * {@inheritDoc}
     */
    public SortedSet<String> getWorkflow() {
      return pWorkflow;
    }



    /**
     * {@inheritDoc}
     */
    public SortedSet<DN> getWorkflowDNs() {
      SortedSet<String> values = getWorkflow();
      SortedSet<DN> dnValues = new TreeSet<DN>();
      for (String value : values) {
        DN dn = INSTANCE.getWorkflowPropertyDefinition().getChildDN(value);
        dnValues.add(dn);
      }
      return dnValues;
    }



    /**
     * {@inheritDoc}
     */
    public Class<? extends NetworkGroupCfg> configurationClass() {
      return NetworkGroupCfg.class;
    }



    /**
     * {@inheritDoc}
     */
    public DN dn() {
      return impl.getDN();
    }

  }
}
