/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.meta;



import org.opends.server.admin.AdministratorAction;
import org.opends.server.admin.AliasDefaultBehaviorProvider;
import org.opends.server.admin.client.AuthorizationException;
import org.opends.server.admin.client.CommunicationException;
import org.opends.server.admin.client.ConcurrentModificationException;
import org.opends.server.admin.client.ManagedObject;
import org.opends.server.admin.client.MissingMandatoryPropertiesException;
import org.opends.server.admin.client.OperationRejectedException;
import org.opends.server.admin.DefaultBehaviorProvider;
import org.opends.server.admin.DefinedDefaultBehaviorProvider;
import org.opends.server.admin.DurationPropertyDefinition;
import org.opends.server.admin.EnumPropertyDefinition;
import org.opends.server.admin.IntegerPropertyDefinition;
import org.opends.server.admin.ManagedObjectAlreadyExistsException;
import org.opends.server.admin.ManagedObjectDefinition;
import org.opends.server.admin.PropertyProvider;
import org.opends.server.admin.server.ConfigurationChangeListener;
import org.opends.server.admin.server.ServerManagedObject;
import org.opends.server.admin.std.client.NetworkGroupResourceLimitsCfgClient;
import org.opends.server.admin.std.server.NetworkGroupResourceLimitsCfg;
import org.opends.server.admin.Tag;
import org.opends.server.admin.TopCfgDefn;
import org.opends.server.types.DN;



/**
 * An interface for querying the Network Group Resource Limits managed
 * object definition meta information.
 * <p>
 * The Network Group Resource Limits are used to define resource
 * limits enforced by the network group.
 */
public final class NetworkGroupResourceLimitsCfgDefn extends ManagedObjectDefinition<NetworkGroupResourceLimitsCfgClient, NetworkGroupResourceLimitsCfg> {

  // The singleton configuration definition instance.
  private static final NetworkGroupResourceLimitsCfgDefn INSTANCE = new NetworkGroupResourceLimitsCfgDefn();



  /**
   * Defines the set of permissable values for the "referral-bind-policy" property.
   * <p>
   * Specifies the referral bind policy.
   * <p>
   * The referral bind policy is used only when the referral policy is
   * set to "follow". It specifies whether the referral is followed
   * with the client bind DN and password or anonymously.
   */
  public static enum ReferralBindPolicy {

    /**
     * Follow referrals anonymously.
     */
    ANONYMOUS("anonymous"),



    /**
     * Follow referrals with the client bind DN and password.
     */
    USE_PASSWORD("use-password");



    // String representation of the value.
    private final String name;



    // Private constructor.
    private ReferralBindPolicy(String name) { this.name = name; }



    /**
     * {@inheritDoc}
     */
    public String toString() { return name; }

  }



  /**
   * Defines the set of permissable values for the "referral-policy" property.
   * <p>
   * Specifies the referral policy.
   * <p>
   * The referral policy defines the behavior when a referral is
   * received. The server can either discard the referral, forward the
   * referral to the client or follow the referral.
   */
  public static enum ReferralPolicy {

    /**
     * Discard referrals.
     */
    DISCARD("discard"),



    /**
     * Follow referrals
     */
    FOLLOW("follow"),



    /**
     * Forward referrals.
     */
    FORWARD("forward");



    // String representation of the value.
    private final String name;



    // Private constructor.
    private ReferralPolicy(String name) { this.name = name; }



    /**
     * {@inheritDoc}
     */
    public String toString() { return name; }

  }



  // The "max-concurrent-ops-per-connection" property definition.
  private static final IntegerPropertyDefinition PD_MAX_CONCURRENT_OPS_PER_CONNECTION;



  // The "max-connections" property definition.
  private static final IntegerPropertyDefinition PD_MAX_CONNECTIONS;



  // The "max-connections-from-same-ip" property definition.
  private static final IntegerPropertyDefinition PD_MAX_CONNECTIONS_FROM_SAME_IP;



  // The "max-ops-per-connection" property definition.
  private static final IntegerPropertyDefinition PD_MAX_OPS_PER_CONNECTION;



  // The "min-substring-length" property definition.
  private static final IntegerPropertyDefinition PD_MIN_SUBSTRING_LENGTH;



  // The "referral-bind-policy" property definition.
  private static final EnumPropertyDefinition<ReferralBindPolicy> PD_REFERRAL_BIND_POLICY;



  // The "referral-hop-limit" property definition.
  private static final IntegerPropertyDefinition PD_REFERRAL_HOP_LIMIT;



  // The "referral-policy" property definition.
  private static final EnumPropertyDefinition<ReferralPolicy> PD_REFERRAL_POLICY;



  // The "search-size-limit" property definition.
  private static final IntegerPropertyDefinition PD_SEARCH_SIZE_LIMIT;



  // The "search-time-limit" property definition.
  private static final DurationPropertyDefinition PD_SEARCH_TIME_LIMIT;



  // Build the "max-concurrent-ops-per-connection" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "max-concurrent-ops-per-connection");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "max-concurrent-ops-per-connection"));
      DefaultBehaviorProvider<Integer> provider = new DefinedDefaultBehaviorProvider<Integer>("0");
      builder.setDefaultBehaviorProvider(provider);
      builder.setLowerLimit(0);
      PD_MAX_CONCURRENT_OPS_PER_CONNECTION = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_MAX_CONCURRENT_OPS_PER_CONNECTION);
  }



  // Build the "max-connections" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "max-connections");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "max-connections"));
      DefaultBehaviorProvider<Integer> provider = new DefinedDefaultBehaviorProvider<Integer>("0");
      builder.setDefaultBehaviorProvider(provider);
      builder.setLowerLimit(0);
      PD_MAX_CONNECTIONS = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_MAX_CONNECTIONS);
  }



  // Build the "max-connections-from-same-ip" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "max-connections-from-same-ip");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "max-connections-from-same-ip"));
      DefaultBehaviorProvider<Integer> provider = new DefinedDefaultBehaviorProvider<Integer>("0");
      builder.setDefaultBehaviorProvider(provider);
      builder.setLowerLimit(0);
      PD_MAX_CONNECTIONS_FROM_SAME_IP = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_MAX_CONNECTIONS_FROM_SAME_IP);
  }



  // Build the "max-ops-per-connection" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "max-ops-per-connection");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "max-ops-per-connection"));
      DefaultBehaviorProvider<Integer> provider = new DefinedDefaultBehaviorProvider<Integer>("0");
      builder.setDefaultBehaviorProvider(provider);
      builder.setLowerLimit(0);
      PD_MAX_OPS_PER_CONNECTION = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_MAX_OPS_PER_CONNECTION);
  }



  // Build the "min-substring-length" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "min-substring-length");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "min-substring-length"));
      DefaultBehaviorProvider<Integer> provider = new DefinedDefaultBehaviorProvider<Integer>("0");
      builder.setDefaultBehaviorProvider(provider);
      builder.setLowerLimit(0);
      PD_MIN_SUBSTRING_LENGTH = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_MIN_SUBSTRING_LENGTH);
  }



  // Build the "referral-bind-policy" property definition.
  static {
      EnumPropertyDefinition.Builder<ReferralBindPolicy> builder = EnumPropertyDefinition.createBuilder(INSTANCE, "referral-bind-policy");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "referral-bind-policy"));
      DefaultBehaviorProvider<ReferralBindPolicy> provider = new DefinedDefaultBehaviorProvider<ReferralBindPolicy>("anonymous");
      builder.setDefaultBehaviorProvider(provider);
      builder.setEnumClass(ReferralBindPolicy.class);
      PD_REFERRAL_BIND_POLICY = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_REFERRAL_BIND_POLICY);
  }



  // Build the "referral-hop-limit" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "referral-hop-limit");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "referral-hop-limit"));
      DefaultBehaviorProvider<Integer> provider = new DefinedDefaultBehaviorProvider<Integer>("0");
      builder.setDefaultBehaviorProvider(provider);
      builder.setLowerLimit(0);
      PD_REFERRAL_HOP_LIMIT = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_REFERRAL_HOP_LIMIT);
  }



  // Build the "referral-policy" property definition.
  static {
      EnumPropertyDefinition.Builder<ReferralPolicy> builder = EnumPropertyDefinition.createBuilder(INSTANCE, "referral-policy");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "referral-policy"));
      DefaultBehaviorProvider<ReferralPolicy> provider = new DefinedDefaultBehaviorProvider<ReferralPolicy>("forward");
      builder.setDefaultBehaviorProvider(provider);
      builder.setEnumClass(ReferralPolicy.class);
      PD_REFERRAL_POLICY = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_REFERRAL_POLICY);
  }



  // Build the "search-size-limit" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "search-size-limit");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "search-size-limit"));
      builder.setDefaultBehaviorProvider(new AliasDefaultBehaviorProvider<Integer>(INSTANCE, "search-size-limit"));
      builder.setLowerLimit(0);
      PD_SEARCH_SIZE_LIMIT = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_SEARCH_SIZE_LIMIT);
  }



  // Build the "search-time-limit" property definition.
  static {
      DurationPropertyDefinition.Builder builder = DurationPropertyDefinition.createBuilder(INSTANCE, "search-time-limit");
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "search-time-limit"));
      builder.setDefaultBehaviorProvider(new AliasDefaultBehaviorProvider<Long>(INSTANCE, "search-time-limit"));
      builder.setBaseUnit("s");
      builder.setLowerLimit("0");
      PD_SEARCH_TIME_LIMIT = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_SEARCH_TIME_LIMIT);
  }



  // Register the tags associated with this managed object definition.
  static {
    INSTANCE.registerTag(Tag.valueOf("core-server"));
  }



  /**
   * Get the Network Group Resource Limits configuration definition
   * singleton.
   *
   * @return Returns the Network Group Resource Limits configuration
   *         definition singleton.
   */
  public static NetworkGroupResourceLimitsCfgDefn getInstance() {
    return INSTANCE;
  }



  /**
   * Private constructor.
   */
  private NetworkGroupResourceLimitsCfgDefn() {
    super("network-group-resource-limits", TopCfgDefn.getInstance());
  }



  /**
   * {@inheritDoc}
   */
  public NetworkGroupResourceLimitsCfgClient createClientConfiguration(
      ManagedObject<? extends NetworkGroupResourceLimitsCfgClient> impl) {
    return new NetworkGroupResourceLimitsCfgClientImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public NetworkGroupResourceLimitsCfg createServerConfiguration(
      ServerManagedObject<? extends NetworkGroupResourceLimitsCfg> impl) {
    return new NetworkGroupResourceLimitsCfgServerImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public Class<NetworkGroupResourceLimitsCfg> getServerConfigurationClass() {
    return NetworkGroupResourceLimitsCfg.class;
  }



  /**
   * Get the "max-concurrent-ops-per-connection" property definition.
   * <p>
   * Specifies the maximum number of concurrent operations per
   * connection.
   * <p>
   * A value of 0 means that no limit is enforced.
   *
   * @return Returns the "max-concurrent-ops-per-connection" property definition.
   */
  public IntegerPropertyDefinition getMaxConcurrentOpsPerConnectionPropertyDefinition() {
    return PD_MAX_CONCURRENT_OPS_PER_CONNECTION;
  }



  /**
   * Get the "max-connections" property definition.
   * <p>
   * Specifies the maximum number of connections in the network group.
   * <p>
   * A value of 0 means that no limit is enforced.
   *
   * @return Returns the "max-connections" property definition.
   */
  public IntegerPropertyDefinition getMaxConnectionsPropertyDefinition() {
    return PD_MAX_CONNECTIONS;
  }



  /**
   * Get the "max-connections-from-same-ip" property definition.
   * <p>
   * Specifies the maximum number of connections from the same client
   * (identified by its IP address).
   * <p>
   * A value of 0 means that no limit is enforced.
   *
   * @return Returns the "max-connections-from-same-ip" property definition.
   */
  public IntegerPropertyDefinition getMaxConnectionsFromSameIPPropertyDefinition() {
    return PD_MAX_CONNECTIONS_FROM_SAME_IP;
  }



  /**
   * Get the "max-ops-per-connection" property definition.
   * <p>
   * Specifies the maximum number of operations per connection.
   * <p>
   * A value of 0 means that no limit is enforced.
   *
   * @return Returns the "max-ops-per-connection" property definition.
   */
  public IntegerPropertyDefinition getMaxOpsPerConnectionPropertyDefinition() {
    return PD_MAX_OPS_PER_CONNECTION;
  }



  /**
   * Get the "min-substring-length" property definition.
   * <p>
   * Specifies the minimum length for a search substring.
   * <p>
   * Search operations with short search substring are likely to match
   * a high number of entries and might degrade performances. A value
   * of 0 indicates that no limit is enforced.
   *
   * @return Returns the "min-substring-length" property definition.
   */
  public IntegerPropertyDefinition getMinSubstringLengthPropertyDefinition() {
    return PD_MIN_SUBSTRING_LENGTH;
  }



  /**
   * Get the "referral-bind-policy" property definition.
   * <p>
   * Specifies the referral bind policy.
   * <p>
   * The referral bind policy is used only when the referral policy is
   * set to "follow". It specifies whether the referral is followed
   * with the client bind DN and password or anonymously.
   *
   * @return Returns the "referral-bind-policy" property definition.
   */
  public EnumPropertyDefinition<ReferralBindPolicy> getReferralBindPolicyPropertyDefinition() {
    return PD_REFERRAL_BIND_POLICY;
  }



  /**
   * Get the "referral-hop-limit" property definition.
   * <p>
   * Specifies the maximum number of hops when following referrals.
   * <p>
   * A value of 0 means that no limit is enforced.
   *
   * @return Returns the "referral-hop-limit" property definition.
   */
  public IntegerPropertyDefinition getReferralHopLimitPropertyDefinition() {
    return PD_REFERRAL_HOP_LIMIT;
  }



  /**
   * Get the "referral-policy" property definition.
   * <p>
   * Specifies the referral policy.
   * <p>
   * The referral policy defines the behavior when a referral is
   * received. The server can either discard the referral, forward the
   * referral to the client or follow the referral.
   *
   * @return Returns the "referral-policy" property definition.
   */
  public EnumPropertyDefinition<ReferralPolicy> getReferralPolicyPropertyDefinition() {
    return PD_REFERRAL_POLICY;
  }



  /**
   * Get the "search-size-limit" property definition.
   * <p>
   * Specifies the maximum number of entries that the Directory Server
   * should return to the client during a search operation.
   * <p>
   * A value of 0 indicates that no size limit is enforced. Note that
   * this is the network-group limit, but it may be overridden on a
   * per-user basis.
   *
   * @return Returns the "search-size-limit" property definition.
   */
  public IntegerPropertyDefinition getSearchSizeLimitPropertyDefinition() {
    return PD_SEARCH_SIZE_LIMIT;
  }



  /**
   * Get the "search-time-limit" property definition.
   * <p>
   * Specifies the maximum length of time that the Directory Server
   * should spend processing a search operation.
   * <p>
   * A value of 0 seconds indicates that no time limit is enforced.
   * Note that this is the network group time limit, but it may be
   * overridden on a per-user basis.
   *
   * @return Returns the "search-time-limit" property definition.
   */
  public DurationPropertyDefinition getSearchTimeLimitPropertyDefinition() {
    return PD_SEARCH_TIME_LIMIT;
  }



  /**
   * Managed object client implementation.
   */
  private static class NetworkGroupResourceLimitsCfgClientImpl implements
    NetworkGroupResourceLimitsCfgClient {

    // Private implementation.
    private ManagedObject<? extends NetworkGroupResourceLimitsCfgClient> impl;



    // Private constructor.
    private NetworkGroupResourceLimitsCfgClientImpl(
        ManagedObject<? extends NetworkGroupResourceLimitsCfgClient> impl) {
      this.impl = impl;
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxConcurrentOpsPerConnection() {
      return impl.getPropertyValue(INSTANCE.getMaxConcurrentOpsPerConnectionPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setMaxConcurrentOpsPerConnection(Integer value) {
      impl.setPropertyValue(INSTANCE.getMaxConcurrentOpsPerConnectionPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxConnections() {
      return impl.getPropertyValue(INSTANCE.getMaxConnectionsPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setMaxConnections(Integer value) {
      impl.setPropertyValue(INSTANCE.getMaxConnectionsPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxConnectionsFromSameIP() {
      return impl.getPropertyValue(INSTANCE.getMaxConnectionsFromSameIPPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setMaxConnectionsFromSameIP(Integer value) {
      impl.setPropertyValue(INSTANCE.getMaxConnectionsFromSameIPPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxOpsPerConnection() {
      return impl.getPropertyValue(INSTANCE.getMaxOpsPerConnectionPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setMaxOpsPerConnection(Integer value) {
      impl.setPropertyValue(INSTANCE.getMaxOpsPerConnectionPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public int getMinSubstringLength() {
      return impl.getPropertyValue(INSTANCE.getMinSubstringLengthPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setMinSubstringLength(Integer value) {
      impl.setPropertyValue(INSTANCE.getMinSubstringLengthPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public ReferralBindPolicy getReferralBindPolicy() {
      return impl.getPropertyValue(INSTANCE.getReferralBindPolicyPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setReferralBindPolicy(ReferralBindPolicy value) {
      impl.setPropertyValue(INSTANCE.getReferralBindPolicyPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public int getReferralHopLimit() {
      return impl.getPropertyValue(INSTANCE.getReferralHopLimitPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setReferralHopLimit(Integer value) {
      impl.setPropertyValue(INSTANCE.getReferralHopLimitPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public ReferralPolicy getReferralPolicy() {
      return impl.getPropertyValue(INSTANCE.getReferralPolicyPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setReferralPolicy(ReferralPolicy value) {
      impl.setPropertyValue(INSTANCE.getReferralPolicyPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public Integer getSearchSizeLimit() {
      return impl.getPropertyValue(INSTANCE.getSearchSizeLimitPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setSearchSizeLimit(Integer value) {
      impl.setPropertyValue(INSTANCE.getSearchSizeLimitPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public Long getSearchTimeLimit() {
      return impl.getPropertyValue(INSTANCE.getSearchTimeLimitPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setSearchTimeLimit(Long value) {
      impl.setPropertyValue(INSTANCE.getSearchTimeLimitPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public ManagedObjectDefinition<? extends NetworkGroupResourceLimitsCfgClient, ? extends NetworkGroupResourceLimitsCfg> definition() {
      return INSTANCE;
    }



    /**
     * {@inheritDoc}
     */
    public PropertyProvider properties() {
      return impl;
    }



    /**
     * {@inheritDoc}
     */
    public void commit() throws ManagedObjectAlreadyExistsException,
        MissingMandatoryPropertiesException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException,
        CommunicationException {
      impl.commit();
    }

  }



  /**
   * Managed object server implementation.
   */
  private static class NetworkGroupResourceLimitsCfgServerImpl implements
    NetworkGroupResourceLimitsCfg {

    // Private implementation.
    private ServerManagedObject<? extends NetworkGroupResourceLimitsCfg> impl;

    // The value of the "max-concurrent-ops-per-connection" property.
    private final int pMaxConcurrentOpsPerConnection;

    // The value of the "max-connections" property.
    private final int pMaxConnections;

    // The value of the "max-connections-from-same-ip" property.
    private final int pMaxConnectionsFromSameIP;

    // The value of the "max-ops-per-connection" property.
    private final int pMaxOpsPerConnection;

    // The value of the "min-substring-length" property.
    private final int pMinSubstringLength;

    // The value of the "referral-bind-policy" property.
    private final ReferralBindPolicy pReferralBindPolicy;

    // The value of the "referral-hop-limit" property.
    private final int pReferralHopLimit;

    // The value of the "referral-policy" property.
    private final ReferralPolicy pReferralPolicy;

    // The value of the "search-size-limit" property.
    private final Integer pSearchSizeLimit;

    // The value of the "search-time-limit" property.
    private final Long pSearchTimeLimit;



    // Private constructor.
    private NetworkGroupResourceLimitsCfgServerImpl(ServerManagedObject<? extends NetworkGroupResourceLimitsCfg> impl) {
      this.impl = impl;
      this.pMaxConcurrentOpsPerConnection = impl.getPropertyValue(INSTANCE.getMaxConcurrentOpsPerConnectionPropertyDefinition());
      this.pMaxConnections = impl.getPropertyValue(INSTANCE.getMaxConnectionsPropertyDefinition());
      this.pMaxConnectionsFromSameIP = impl.getPropertyValue(INSTANCE.getMaxConnectionsFromSameIPPropertyDefinition());
      this.pMaxOpsPerConnection = impl.getPropertyValue(INSTANCE.getMaxOpsPerConnectionPropertyDefinition());
      this.pMinSubstringLength = impl.getPropertyValue(INSTANCE.getMinSubstringLengthPropertyDefinition());
      this.pReferralBindPolicy = impl.getPropertyValue(INSTANCE.getReferralBindPolicyPropertyDefinition());
      this.pReferralHopLimit = impl.getPropertyValue(INSTANCE.getReferralHopLimitPropertyDefinition());
      this.pReferralPolicy = impl.getPropertyValue(INSTANCE.getReferralPolicyPropertyDefinition());
      this.pSearchSizeLimit = impl.getPropertyValue(INSTANCE.getSearchSizeLimitPropertyDefinition());
      this.pSearchTimeLimit = impl.getPropertyValue(INSTANCE.getSearchTimeLimitPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void addChangeListener(
        ConfigurationChangeListener<NetworkGroupResourceLimitsCfg> listener) {
      impl.registerChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeChangeListener(
        ConfigurationChangeListener<NetworkGroupResourceLimitsCfg> listener) {
      impl.deregisterChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxConcurrentOpsPerConnection() {
      return pMaxConcurrentOpsPerConnection;
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxConnections() {
      return pMaxConnections;
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxConnectionsFromSameIP() {
      return pMaxConnectionsFromSameIP;
    }



    /**
     * {@inheritDoc}
     */
    public int getMaxOpsPerConnection() {
      return pMaxOpsPerConnection;
    }



    /**
     * {@inheritDoc}
     */
    public int getMinSubstringLength() {
      return pMinSubstringLength;
    }



    /**
     * {@inheritDoc}
     */
    public ReferralBindPolicy getReferralBindPolicy() {
      return pReferralBindPolicy;
    }



    /**
     * {@inheritDoc}
     */
    public int getReferralHopLimit() {
      return pReferralHopLimit;
    }



    /**
     * {@inheritDoc}
     */
    public ReferralPolicy getReferralPolicy() {
      return pReferralPolicy;
    }



    /**
     * {@inheritDoc}
     */
    public Integer getSearchSizeLimit() {
      return pSearchSizeLimit;
    }



    /**
     * {@inheritDoc}
     */
    public Long getSearchTimeLimit() {
      return pSearchTimeLimit;
    }



    /**
     * {@inheritDoc}
     */
    public Class<? extends NetworkGroupResourceLimitsCfg> configurationClass() {
      return NetworkGroupResourceLimitsCfg.class;
    }



    /**
     * {@inheritDoc}
     */
    public DN dn() {
      return impl.getDN();
    }

  }
}
