/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.meta;



import java.util.Collection;
import java.util.SortedSet;
import java.util.TreeSet;
import org.opends.server.admin.AdministratorAction;
import org.opends.server.admin.AggregationPropertyDefinition;
import org.opends.server.admin.BooleanPropertyDefinition;
import org.opends.server.admin.client.AuthorizationException;
import org.opends.server.admin.client.CommunicationException;
import org.opends.server.admin.client.ConcurrentModificationException;
import org.opends.server.admin.client.ManagedObject;
import org.opends.server.admin.client.ManagedObjectDecodingException;
import org.opends.server.admin.client.MissingMandatoryPropertiesException;
import org.opends.server.admin.client.OperationRejectedException;
import org.opends.server.admin.condition.Conditions;
import org.opends.server.admin.DefaultBehaviorException;
import org.opends.server.admin.DefinitionDecodingException;
import org.opends.server.admin.IntegerPropertyDefinition;
import org.opends.server.admin.ManagedObjectAlreadyExistsException;
import org.opends.server.admin.ManagedObjectDefinition;
import org.opends.server.admin.ManagedObjectNotFoundException;
import org.opends.server.admin.OptionalRelationDefinition;
import org.opends.server.admin.PropertyIsReadOnlyException;
import org.opends.server.admin.PropertyOption;
import org.opends.server.admin.PropertyProvider;
import org.opends.server.admin.server.ConfigurationAddListener;
import org.opends.server.admin.server.ConfigurationChangeListener;
import org.opends.server.admin.server.ConfigurationDeleteListener;
import org.opends.server.admin.server.ServerManagedObject;
import org.opends.server.admin.std.client.NetworkGroupCfgClient;
import org.opends.server.admin.std.client.NetworkGroupCriteriaCfgClient;
import org.opends.server.admin.std.client.NetworkGroupRequestFilteringPolicyCfgClient;
import org.opends.server.admin.std.client.NetworkGroupResourceLimitsCfgClient;
import org.opends.server.admin.std.client.WorkflowCfgClient;
import org.opends.server.admin.std.server.NetworkGroupCfg;
import org.opends.server.admin.std.server.NetworkGroupCriteriaCfg;
import org.opends.server.admin.std.server.NetworkGroupRequestFilteringPolicyCfg;
import org.opends.server.admin.std.server.NetworkGroupResourceLimitsCfg;
import org.opends.server.admin.std.server.WorkflowCfg;
import org.opends.server.admin.StringPropertyDefinition;
import org.opends.server.admin.Tag;
import org.opends.server.admin.TopCfgDefn;
import org.opends.server.admin.UndefinedDefaultBehaviorProvider;
import org.opends.server.config.ConfigException;
import org.opends.server.types.DN;



/**
 * An interface for querying the Network Group managed object
 * definition meta information.
 * <p>
 * The Network Group is used to classify incoming connections and
 * route requests to workflows.
 */
public final class NetworkGroupCfgDefn extends ManagedObjectDefinition<NetworkGroupCfgClient, NetworkGroupCfg> {

  // The singleton configuration definition instance.
  private static final NetworkGroupCfgDefn INSTANCE = new NetworkGroupCfgDefn();



  // The "enabled" property definition.
  private static final BooleanPropertyDefinition PD_ENABLED;



  // The "network-group-id" property definition.
  private static final StringPropertyDefinition PD_NETWORK_GROUP_ID;



  // The "priority" property definition.
  private static final IntegerPropertyDefinition PD_PRIORITY;



  // The "workflow" property definition.
  private static final AggregationPropertyDefinition<WorkflowCfgClient, WorkflowCfg> PD_WORKFLOW;



  // The "network-group-criteria" relation definition.
  private static final OptionalRelationDefinition<NetworkGroupCriteriaCfgClient, NetworkGroupCriteriaCfg> RD_NETWORK_GROUP_CRITERIA;



  // The "network-group-request-filtering-policy" relation definition.
  private static final OptionalRelationDefinition<NetworkGroupRequestFilteringPolicyCfgClient, NetworkGroupRequestFilteringPolicyCfg> RD_NETWORK_GROUP_REQUEST_FILTERING_POLICY;



  // The "network-group-resource-limits" relation definition.
  private static final OptionalRelationDefinition<NetworkGroupResourceLimitsCfgClient, NetworkGroupResourceLimitsCfg> RD_NETWORK_GROUP_RESOURCE_LIMITS;



  // Build the "enabled" property definition.
  static {
      BooleanPropertyDefinition.Builder builder = BooleanPropertyDefinition.createBuilder(INSTANCE, "enabled");
      builder.setOption(PropertyOption.MANDATORY);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "enabled"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<Boolean>());
      PD_ENABLED = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_ENABLED);
  }



  // Build the "network-group-id" property definition.
  static {
      StringPropertyDefinition.Builder builder = StringPropertyDefinition.createBuilder(INSTANCE, "network-group-id");
      builder.setOption(PropertyOption.READ_ONLY);
      builder.setOption(PropertyOption.MANDATORY);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "network-group-id"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<String>());
      PD_NETWORK_GROUP_ID = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_NETWORK_GROUP_ID);
  }



  // Build the "priority" property definition.
  static {
      IntegerPropertyDefinition.Builder builder = IntegerPropertyDefinition.createBuilder(INSTANCE, "priority");
      builder.setOption(PropertyOption.MANDATORY);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "priority"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<Integer>());
      builder.setLowerLimit(0);
      PD_PRIORITY = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_PRIORITY);
  }



  // Build the "workflow" property definition.
  static {
      AggregationPropertyDefinition.Builder<WorkflowCfgClient, WorkflowCfg> builder = AggregationPropertyDefinition.createBuilder(INSTANCE, "workflow");
      builder.setOption(PropertyOption.MULTI_VALUED);
      builder.setAdministratorAction(new AdministratorAction(AdministratorAction.Type.NONE, INSTANCE, "workflow"));
      builder.setDefaultBehaviorProvider(new UndefinedDefaultBehaviorProvider<String>());
      builder.setParentPath("/");
      builder.setRelationDefinition("workflow");
      builder.setTargetIsEnabledCondition(Conditions.contains("enabled", "true"));
      PD_WORKFLOW = builder.getInstance();
      INSTANCE.registerPropertyDefinition(PD_WORKFLOW);
      INSTANCE.registerConstraint(PD_WORKFLOW.getSourceConstraint());
  }



  // Build the "network-group-criteria" relation definition.
  static {
    OptionalRelationDefinition.Builder<NetworkGroupCriteriaCfgClient, NetworkGroupCriteriaCfg> builder =
      new OptionalRelationDefinition.Builder<NetworkGroupCriteriaCfgClient, NetworkGroupCriteriaCfg>(INSTANCE, "network-group-criteria", NetworkGroupCriteriaCfgDefn.getInstance());
    RD_NETWORK_GROUP_CRITERIA = builder.getInstance();
    INSTANCE.registerRelationDefinition(RD_NETWORK_GROUP_CRITERIA);
  }



  // Build the "network-group-request-filtering-policy" relation definition.
  static {
    OptionalRelationDefinition.Builder<NetworkGroupRequestFilteringPolicyCfgClient, NetworkGroupRequestFilteringPolicyCfg> builder =
      new OptionalRelationDefinition.Builder<NetworkGroupRequestFilteringPolicyCfgClient, NetworkGroupRequestFilteringPolicyCfg>(INSTANCE, "network-group-request-filtering-policy", NetworkGroupRequestFilteringPolicyCfgDefn.getInstance());
    RD_NETWORK_GROUP_REQUEST_FILTERING_POLICY = builder.getInstance();
    INSTANCE.registerRelationDefinition(RD_NETWORK_GROUP_REQUEST_FILTERING_POLICY);
  }



  // Build the "network-group-resource-limits" relation definition.
  static {
    OptionalRelationDefinition.Builder<NetworkGroupResourceLimitsCfgClient, NetworkGroupResourceLimitsCfg> builder =
      new OptionalRelationDefinition.Builder<NetworkGroupResourceLimitsCfgClient, NetworkGroupResourceLimitsCfg>(INSTANCE, "network-group-resource-limits", NetworkGroupResourceLimitsCfgDefn.getInstance());
    RD_NETWORK_GROUP_RESOURCE_LIMITS = builder.getInstance();
    INSTANCE.registerRelationDefinition(RD_NETWORK_GROUP_RESOURCE_LIMITS);
  }



  // Register the tags associated with this managed object definition.
  static {
    INSTANCE.registerTag(Tag.valueOf("core-server"));
  }



  /**
   * Get the Network Group configuration definition singleton.
   *
   * @return Returns the Network Group configuration definition
   *         singleton.
   */
  public static NetworkGroupCfgDefn getInstance() {
    return INSTANCE;
  }



  /**
   * Private constructor.
   */
  private NetworkGroupCfgDefn() {
    super("network-group", TopCfgDefn.getInstance());
  }



  /**
   * {@inheritDoc}
   */
  public NetworkGroupCfgClient createClientConfiguration(
      ManagedObject<? extends NetworkGroupCfgClient> impl) {
    return new NetworkGroupCfgClientImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public NetworkGroupCfg createServerConfiguration(
      ServerManagedObject<? extends NetworkGroupCfg> impl) {
    return new NetworkGroupCfgServerImpl(impl);
  }



  /**
   * {@inheritDoc}
   */
  public Class<NetworkGroupCfg> getServerConfigurationClass() {
    return NetworkGroupCfg.class;
  }



  /**
   * Get the "enabled" property definition.
   * <p>
   * Indicates whether the Network Group is enabled for use in the
   * server.
   * <p>
   * If a network group is not enabled, its workflows will not be
   * accessible when processing operations.
   *
   * @return Returns the "enabled" property definition.
   */
  public BooleanPropertyDefinition getEnabledPropertyDefinition() {
    return PD_ENABLED;
  }



  /**
   * Get the "network-group-id" property definition.
   * <p>
   * Specifies the name that is used to identify the associated
   * Network Group .
   * <p>
   * The name must be unique among all the Network Groups in the
   * server.
   *
   * @return Returns the "network-group-id" property definition.
   */
  public StringPropertyDefinition getNetworkGroupIdPropertyDefinition() {
    return PD_NETWORK_GROUP_ID;
  }



  /**
   * Get the "priority" property definition.
   * <p>
   * Specifies the order in which the network groups are evaluated.
   * <p>
   * A client connection is first compared against network group with
   * priority 1. If the client connection does not match the network
   * group criteria, the client connection is compared against network
   * group with priority 2 etc...
   *
   * @return Returns the "priority" property definition.
   */
  public IntegerPropertyDefinition getPriorityPropertyDefinition() {
    return PD_PRIORITY;
  }



  /**
   * Get the "workflow" property definition.
   * <p>
   * Identifies the workflows in the network group.
   *
   * @return Returns the "workflow" property definition.
   */
  public AggregationPropertyDefinition<WorkflowCfgClient, WorkflowCfg> getWorkflowPropertyDefinition() {
    return PD_WORKFLOW;
  }



  /**
   * Get the "network-group-criteria" relation definition.
   *
   * @return Returns the "network-group-criteria" relation definition.
   */
  public OptionalRelationDefinition<NetworkGroupCriteriaCfgClient,NetworkGroupCriteriaCfg> getNetworkGroupCriteriaRelationDefinition() {
    return RD_NETWORK_GROUP_CRITERIA;
  }



  /**
   * Get the "network-group-request-filtering-policy" relation definition.
   *
   * @return Returns the "network-group-request-filtering-policy" relation definition.
   */
  public OptionalRelationDefinition<NetworkGroupRequestFilteringPolicyCfgClient,NetworkGroupRequestFilteringPolicyCfg> getNetworkGroupRequestFilteringPolicyRelationDefinition() {
    return RD_NETWORK_GROUP_REQUEST_FILTERING_POLICY;
  }



  /**
   * Get the "network-group-resource-limits" relation definition.
   *
   * @return Returns the "network-group-resource-limits" relation definition.
   */
  public OptionalRelationDefinition<NetworkGroupResourceLimitsCfgClient,NetworkGroupResourceLimitsCfg> getNetworkGroupResourceLimitsRelationDefinition() {
    return RD_NETWORK_GROUP_RESOURCE_LIMITS;
  }



  /**
   * Managed object client implementation.
   */
  private static class NetworkGroupCfgClientImpl implements
    NetworkGroupCfgClient {

    // Private implementation.
    private ManagedObject<? extends NetworkGroupCfgClient> impl;



    // Private constructor.
    private NetworkGroupCfgClientImpl(
        ManagedObject<? extends NetworkGroupCfgClient> impl) {
      this.impl = impl;
    }



    /**
     * {@inheritDoc}
     */
    public Boolean isEnabled() {
      return impl.getPropertyValue(INSTANCE.getEnabledPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setEnabled(boolean value) {
      impl.setPropertyValue(INSTANCE.getEnabledPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public String getNetworkGroupId() {
      return impl.getPropertyValue(INSTANCE.getNetworkGroupIdPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setNetworkGroupId(String value) throws PropertyIsReadOnlyException {
      impl.setPropertyValue(INSTANCE.getNetworkGroupIdPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public Integer getPriority() {
      return impl.getPropertyValue(INSTANCE.getPriorityPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setPriority(int value) {
      impl.setPropertyValue(INSTANCE.getPriorityPropertyDefinition(), value);
    }



    /**
     * {@inheritDoc}
     */
    public SortedSet<String> getWorkflow() {
      return impl.getPropertyValues(INSTANCE.getWorkflowPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void setWorkflow(Collection<String> values) {
      impl.setPropertyValues(INSTANCE.getWorkflowPropertyDefinition(), values);
    }



    /**
     * {@inheritDoc}
     */
    public boolean hasNetworkGroupCriteria() throws ConcurrentModificationException,
        AuthorizationException, CommunicationException {
      return impl.hasChild(INSTANCE.getNetworkGroupCriteriaRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public NetworkGroupCriteriaCfgClient getNetworkGroupCriteria()
        throws DefinitionDecodingException, ManagedObjectDecodingException,
        ManagedObjectNotFoundException, ConcurrentModificationException,
        AuthorizationException, CommunicationException {
      return impl.getChild(INSTANCE.getNetworkGroupCriteriaRelationDefinition()).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public <M extends NetworkGroupCriteriaCfgClient> M createNetworkGroupCriteria(
        ManagedObjectDefinition<M, ? extends NetworkGroupCriteriaCfg> d, Collection<DefaultBehaviorException> exceptions) {
      return impl.createChild(INSTANCE.getNetworkGroupCriteriaRelationDefinition(), d, exceptions).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupCriteria()
        throws ManagedObjectNotFoundException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException, CommunicationException {
      impl.removeChild(INSTANCE.getNetworkGroupCriteriaRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public boolean hasNetworkGroupRequestFilteringPolicy() throws ConcurrentModificationException,
        AuthorizationException, CommunicationException {
      return impl.hasChild(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public NetworkGroupRequestFilteringPolicyCfgClient getNetworkGroupRequestFilteringPolicy()
        throws DefinitionDecodingException, ManagedObjectDecodingException,
        ManagedObjectNotFoundException, ConcurrentModificationException,
        AuthorizationException, CommunicationException {
      return impl.getChild(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition()).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public <M extends NetworkGroupRequestFilteringPolicyCfgClient> M createNetworkGroupRequestFilteringPolicy(
        ManagedObjectDefinition<M, ? extends NetworkGroupRequestFilteringPolicyCfg> d, Collection<DefaultBehaviorException> exceptions) {
      return impl.createChild(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition(), d, exceptions).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupRequestFilteringPolicy()
        throws ManagedObjectNotFoundException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException, CommunicationException {
      impl.removeChild(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public boolean hasNetworkGroupResourceLimits() throws ConcurrentModificationException,
        AuthorizationException, CommunicationException {
      return impl.hasChild(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public NetworkGroupResourceLimitsCfgClient getNetworkGroupResourceLimits()
        throws DefinitionDecodingException, ManagedObjectDecodingException,
        ManagedObjectNotFoundException, ConcurrentModificationException,
        AuthorizationException, CommunicationException {
      return impl.getChild(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition()).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public <M extends NetworkGroupResourceLimitsCfgClient> M createNetworkGroupResourceLimits(
        ManagedObjectDefinition<M, ? extends NetworkGroupResourceLimitsCfg> d, Collection<DefaultBehaviorException> exceptions) {
      return impl.createChild(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition(), d, exceptions).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupResourceLimits()
        throws ManagedObjectNotFoundException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException, CommunicationException {
      impl.removeChild(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public ManagedObjectDefinition<? extends NetworkGroupCfgClient, ? extends NetworkGroupCfg> definition() {
      return INSTANCE;
    }



    /**
     * {@inheritDoc}
     */
    public PropertyProvider properties() {
      return impl;
    }



    /**
     * {@inheritDoc}
     */
    public void commit() throws ManagedObjectAlreadyExistsException,
        MissingMandatoryPropertiesException, ConcurrentModificationException,
        OperationRejectedException, AuthorizationException,
        CommunicationException {
      impl.commit();
    }

  }



  /**
   * Managed object server implementation.
   */
  private static class NetworkGroupCfgServerImpl implements
    NetworkGroupCfg {

    // Private implementation.
    private ServerManagedObject<? extends NetworkGroupCfg> impl;

    // The value of the "enabled" property.
    private final boolean pEnabled;

    // The value of the "network-group-id" property.
    private final String pNetworkGroupId;

    // The value of the "priority" property.
    private final int pPriority;

    // The value of the "workflow" property.
    private final SortedSet<String> pWorkflow;



    // Private constructor.
    private NetworkGroupCfgServerImpl(ServerManagedObject<? extends NetworkGroupCfg> impl) {
      this.impl = impl;
      this.pEnabled = impl.getPropertyValue(INSTANCE.getEnabledPropertyDefinition());
      this.pNetworkGroupId = impl.getPropertyValue(INSTANCE.getNetworkGroupIdPropertyDefinition());
      this.pPriority = impl.getPropertyValue(INSTANCE.getPriorityPropertyDefinition());
      this.pWorkflow = impl.getPropertyValues(INSTANCE.getWorkflowPropertyDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public void addChangeListener(
        ConfigurationChangeListener<NetworkGroupCfg> listener) {
      impl.registerChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeChangeListener(
        ConfigurationChangeListener<NetworkGroupCfg> listener) {
      impl.deregisterChangeListener(listener);
    }



    /**
     * {@inheritDoc}
     */
    public boolean isEnabled() {
      return pEnabled;
    }



    /**
     * {@inheritDoc}
     */
    public String getNetworkGroupId() {
      return pNetworkGroupId;
    }



    /**
     * {@inheritDoc}
     */
    public int getPriority() {
      return pPriority;
    }



    /**
     * {@inheritDoc}
     */
    public SortedSet<String> getWorkflow() {
      return pWorkflow;
    }



    /**
     * {@inheritDoc}
     */
    public SortedSet<DN> getWorkflowDNs() {
      SortedSet<String> values = getWorkflow();
      SortedSet<DN> dnValues = new TreeSet<DN>();
      for (String value : values) {
        DN dn = INSTANCE.getWorkflowPropertyDefinition().getChildDN(value);
        dnValues.add(dn);
      }
      return dnValues;
    }



    /**
     * {@inheritDoc}
     */
    public boolean hasNetworkGroupCriteria() {
      return impl.hasChild(INSTANCE.getNetworkGroupCriteriaRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public NetworkGroupCriteriaCfg getNetworkGroupCriteria() throws ConfigException {
      return impl.getChild(INSTANCE.getNetworkGroupCriteriaRelationDefinition()).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public void addNetworkGroupCriteriaAddListener(
        ConfigurationAddListener<NetworkGroupCriteriaCfg> listener) throws ConfigException {
      impl.registerAddListener(INSTANCE.getNetworkGroupCriteriaRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupCriteriaAddListener(
        ConfigurationAddListener<NetworkGroupCriteriaCfg> listener) {
      impl.deregisterAddListener(INSTANCE.getNetworkGroupCriteriaRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void addNetworkGroupCriteriaDeleteListener(
        ConfigurationDeleteListener<NetworkGroupCriteriaCfg> listener) throws ConfigException {
      impl.registerDeleteListener(INSTANCE.getNetworkGroupCriteriaRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupCriteriaDeleteListener(
        ConfigurationDeleteListener<NetworkGroupCriteriaCfg> listener) {
      impl.deregisterDeleteListener(INSTANCE.getNetworkGroupCriteriaRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public boolean hasNetworkGroupRequestFilteringPolicy() {
      return impl.hasChild(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public NetworkGroupRequestFilteringPolicyCfg getNetworkGroupRequestFilteringPolicy() throws ConfigException {
      return impl.getChild(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition()).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public void addNetworkGroupRequestFilteringPolicyAddListener(
        ConfigurationAddListener<NetworkGroupRequestFilteringPolicyCfg> listener) throws ConfigException {
      impl.registerAddListener(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupRequestFilteringPolicyAddListener(
        ConfigurationAddListener<NetworkGroupRequestFilteringPolicyCfg> listener) {
      impl.deregisterAddListener(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void addNetworkGroupRequestFilteringPolicyDeleteListener(
        ConfigurationDeleteListener<NetworkGroupRequestFilteringPolicyCfg> listener) throws ConfigException {
      impl.registerDeleteListener(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupRequestFilteringPolicyDeleteListener(
        ConfigurationDeleteListener<NetworkGroupRequestFilteringPolicyCfg> listener) {
      impl.deregisterDeleteListener(INSTANCE.getNetworkGroupRequestFilteringPolicyRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public boolean hasNetworkGroupResourceLimits() {
      return impl.hasChild(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition());
    }



    /**
     * {@inheritDoc}
     */
    public NetworkGroupResourceLimitsCfg getNetworkGroupResourceLimits() throws ConfigException {
      return impl.getChild(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition()).getConfiguration();
    }



    /**
     * {@inheritDoc}
     */
    public void addNetworkGroupResourceLimitsAddListener(
        ConfigurationAddListener<NetworkGroupResourceLimitsCfg> listener) throws ConfigException {
      impl.registerAddListener(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupResourceLimitsAddListener(
        ConfigurationAddListener<NetworkGroupResourceLimitsCfg> listener) {
      impl.deregisterAddListener(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void addNetworkGroupResourceLimitsDeleteListener(
        ConfigurationDeleteListener<NetworkGroupResourceLimitsCfg> listener) throws ConfigException {
      impl.registerDeleteListener(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public void removeNetworkGroupResourceLimitsDeleteListener(
        ConfigurationDeleteListener<NetworkGroupResourceLimitsCfg> listener) {
      impl.deregisterDeleteListener(INSTANCE.getNetworkGroupResourceLimitsRelationDefinition(), listener);
    }



    /**
     * {@inheritDoc}
     */
    public Class<? extends NetworkGroupCfg> configurationClass() {
      return NetworkGroupCfg.class;
    }



    /**
     * {@inheritDoc}
     */
    public DN dn() {
      return impl.getDN();
    }

  }
}
