/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.client;



import java.util.Collection;
import java.util.SortedSet;
import org.opends.server.admin.client.AuthorizationException;
import org.opends.server.admin.client.CommunicationException;
import org.opends.server.admin.client.ConcurrentModificationException;
import org.opends.server.admin.client.ManagedObjectDecodingException;
import org.opends.server.admin.client.OperationRejectedException;
import org.opends.server.admin.ConfigurationClient;
import org.opends.server.admin.DefaultBehaviorException;
import org.opends.server.admin.DefinitionDecodingException;
import org.opends.server.admin.IllegalPropertyValueException;
import org.opends.server.admin.ManagedObjectDefinition;
import org.opends.server.admin.ManagedObjectNotFoundException;
import org.opends.server.admin.PropertyIsReadOnlyException;
import org.opends.server.admin.std.server.NetworkGroupCfg;
import org.opends.server.admin.std.server.NetworkGroupCriteriaCfg;
import org.opends.server.admin.std.server.NetworkGroupRequestFilteringPolicyCfg;
import org.opends.server.admin.std.server.NetworkGroupResourceLimitsCfg;



/**
 * A client-side interface for reading and modifying Network Group
 * settings.
 * <p>
 * The Network Group is used to classify incoming connections and
 * route requests to workflows.
 */
public interface NetworkGroupCfgClient extends ConfigurationClient {

  /**
   * Get the configuration definition associated with this Network Group.
   *
   * @return Returns the configuration definition associated with this Network Group.
   */
  ManagedObjectDefinition<? extends NetworkGroupCfgClient, ? extends NetworkGroupCfg> definition();



  /**
   * Gets the "enabled" property.
   * <p>
   * Indicates whether the Network Group is enabled for use in the
   * server.
   * <p>
   * If a network group is not enabled, its workflows will not be
   * accessible when processing operations.
   *
   * @return Returns the value of the "enabled" property.
   */
  Boolean isEnabled();



  /**
   * Sets the "enabled" property.
   * <p>
   * Indicates whether the Network Group is enabled for use in the
   * server.
   * <p>
   * If a network group is not enabled, its workflows will not be
   * accessible when processing operations.
   *
   * @param value The value of the "enabled" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   */
  void setEnabled(boolean value) throws IllegalPropertyValueException;



  /**
   * Gets the "network-group-id" property.
   * <p>
   * Specifies the name that is used to identify the associated
   * Network Group .
   * <p>
   * The name must be unique among all the Network Groups in the
   * server.
   *
   * @return Returns the value of the "network-group-id" property.
   */
  String getNetworkGroupId();



  /**
   * Sets the "network-group-id" property.
   * <p>
   * Specifies the name that is used to identify the associated
   * Network Group .
   * <p>
   * The name must be unique among all the Network Groups in the
   * server.
   * <p>
   * This property is read-only and can only be modified during
   * creation of a Network Group.
   *
   * @param value The value of the "network-group-id" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   * @throws PropertyIsReadOnlyException
   *           If this Network Group is not being initialized.
   */
  void setNetworkGroupId(String value) throws IllegalPropertyValueException, PropertyIsReadOnlyException;



  /**
   * Gets the "priority" property.
   * <p>
   * Specifies the order in which the network groups are evaluated.
   * <p>
   * A client connection is first compared against network group with
   * priority 1. If the client connection does not match the network
   * group criteria, the client connection is compared against network
   * group with priority 2 etc...
   *
   * @return Returns the value of the "priority" property.
   */
  Integer getPriority();



  /**
   * Sets the "priority" property.
   * <p>
   * Specifies the order in which the network groups are evaluated.
   * <p>
   * A client connection is first compared against network group with
   * priority 1. If the client connection does not match the network
   * group criteria, the client connection is compared against network
   * group with priority 2 etc...
   *
   * @param value The value of the "priority" property.
   * @throws IllegalPropertyValueException
   *           If the new value is invalid.
   */
  void setPriority(int value) throws IllegalPropertyValueException;



  /**
   * Gets the "workflow" property.
   * <p>
   * Identifies the workflows in the network group.
   *
   * @return Returns the values of the "workflow" property.
   */
  SortedSet<String> getWorkflow();



  /**
   * Sets the "workflow" property.
   * <p>
   * Identifies the workflows in the network group.
   *
   * @param values The values of the "workflow" property.
   * @throws IllegalPropertyValueException
   *           If one or more of the new values are invalid.
   */
  void setWorkflow(Collection<String> values) throws IllegalPropertyValueException;



  /**
   * Determines whether or not the Network Group Criteria exists.
   *
   * @return Returns <true> if the Network Group Criteria exists.
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to make the determination because
   *           the client does not have the correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  boolean hasNetworkGroupCriteria() throws ConcurrentModificationException,
      AuthorizationException, CommunicationException;



  /**
   * Gets the Network Group Criteria if it is present.
   *
   * @return Returns the Network Group Criteria if it is present.
   * @throws DefinitionDecodingException
   *           If the Network Group Criteria was found but its type
   *           could not be determined.
   * @throws ManagedObjectDecodingException
   *           If the Network Group Criteria was found but one or more
   *           of its properties could not be decoded.
   * @throws ManagedObjectNotFoundException
   *           If the Network Group Criteria is not present.
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to retrieve the Network Group
   *           Criteria because the client does not have the correct
   *           privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  NetworkGroupCriteriaCfgClient getNetworkGroupCriteria()
      throws DefinitionDecodingException, ManagedObjectDecodingException,
      ManagedObjectNotFoundException, ConcurrentModificationException,
      AuthorizationException, CommunicationException;



  /**
   * Creates a new Network Group Criteria. The new Network Group
   * Criteria will initially not contain any property values (including
   * mandatory properties). Once the Network Group Criteria has been
   * configured it can be added to the server using the {@link
   * #commit()} method.
   *
   * @param <C>
   *          The type of the Network Group Criteria being created.
   * @param d
   *          The definition of the Network Group Criteria to be
   *          created.
   * @param exceptions
   *          An optional collection in which to place any {@link
   *          DefaultBehaviorException}s that occurred whilst
   *          attempting to determine the default values of the Network
   *          Group Criteria. This argument can be <code>null<code>.
   * @return Returns a new Network Group Criteria configuration
   *         instance.
   */
  <C extends NetworkGroupCriteriaCfgClient> C createNetworkGroupCriteria(
      ManagedObjectDefinition<C, ? extends NetworkGroupCriteriaCfg> d, Collection<DefaultBehaviorException> exceptions);



  /**
   * Removes the Network Group Criteria if it exists.
   *
   * @throws ManagedObjectNotFoundException
   *           If the Network Group Criteria does not exist.
   * @throws OperationRejectedException
   *           If the server refuses to remove the Network Group
   *           Criteria due to some server-side constraint which cannot
   *           be satisfied (for example, if it is referenced by
   *           another managed object).
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to remove the Network Group
   *           Criteria because the client does not have the correct
   *           privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  void removeNetworkGroupCriteria()
      throws ManagedObjectNotFoundException, OperationRejectedException,
      ConcurrentModificationException, AuthorizationException,
      CommunicationException;



  /**
   * Determines whether or not the Network Group Request Filtering
   * Policy exists.
   *
   * @return Returns <true> if the Network Group Request Filtering
   *         Policy exists.
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to make the determination because
   *           the client does not have the correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  boolean hasNetworkGroupRequestFilteringPolicy() throws ConcurrentModificationException,
      AuthorizationException, CommunicationException;



  /**
   * Gets the Network Group Request Filtering Policy if it is present.
   *
   * @return Returns the Network Group Request Filtering Policy if it
   *         is present.
   * @throws DefinitionDecodingException
   *           If the Network Group Request Filtering Policy was found
   *           but its type could not be determined.
   * @throws ManagedObjectDecodingException
   *           If the Network Group Request Filtering Policy was found
   *           but one or more of its properties could not be decoded.
   * @throws ManagedObjectNotFoundException
   *           If the Network Group Request Filtering Policy is not
   *           present.
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to retrieve the Network Group
   *           Request Filtering Policy because the client does not
   *           have the correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  NetworkGroupRequestFilteringPolicyCfgClient getNetworkGroupRequestFilteringPolicy()
      throws DefinitionDecodingException, ManagedObjectDecodingException,
      ManagedObjectNotFoundException, ConcurrentModificationException,
      AuthorizationException, CommunicationException;



  /**
   * Creates a new Network Group Request Filtering Policy. The new
   * Network Group Request Filtering Policy will initially not contain
   * any property values (including mandatory properties). Once the
   * Network Group Request Filtering Policy has been configured it can
   * be added to the server using the {@link #commit()} method.
   *
   * @param <C>
   *          The type of the Network Group Request Filtering Policy
   *          being created.
   * @param d
   *          The definition of the Network Group Request Filtering
   *          Policy to be created.
   * @param exceptions
   *          An optional collection in which to place any {@link
   *          DefaultBehaviorException}s that occurred whilst
   *          attempting to determine the default values of the Network
   *          Group Request Filtering Policy. This argument can be
   *          <code>null<code>.
   * @return Returns a new Network Group Request Filtering Policy
   *         configuration instance.
   */
  <C extends NetworkGroupRequestFilteringPolicyCfgClient> C createNetworkGroupRequestFilteringPolicy(
      ManagedObjectDefinition<C, ? extends NetworkGroupRequestFilteringPolicyCfg> d, Collection<DefaultBehaviorException> exceptions);



  /**
   * Removes the Network Group Request Filtering Policy if it exists.
   *
   * @throws ManagedObjectNotFoundException
   *           If the Network Group Request Filtering Policy does not
   *           exist.
   * @throws OperationRejectedException
   *           If the server refuses to remove the Network Group
   *           Request Filtering Policy due to some server-side
   *           constraint which cannot be satisfied (for example, if it
   *           is referenced by another managed object).
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to remove the Network Group
   *           Request Filtering Policy because the client does not
   *           have the correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  void removeNetworkGroupRequestFilteringPolicy()
      throws ManagedObjectNotFoundException, OperationRejectedException,
      ConcurrentModificationException, AuthorizationException,
      CommunicationException;



  /**
   * Determines whether or not the Network Group Resource Limits
   * exists.
   *
   * @return Returns <true> if the Network Group Resource Limits
   *         exists.
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to make the determination because
   *           the client does not have the correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  boolean hasNetworkGroupResourceLimits() throws ConcurrentModificationException,
      AuthorizationException, CommunicationException;



  /**
   * Gets the Network Group Resource Limits if it is present.
   *
   * @return Returns the Network Group Resource Limits if it is
   *         present.
   * @throws DefinitionDecodingException
   *           If the Network Group Resource Limits was found but its
   *           type could not be determined.
   * @throws ManagedObjectDecodingException
   *           If the Network Group Resource Limits was found but one
   *           or more of its properties could not be decoded.
   * @throws ManagedObjectNotFoundException
   *           If the Network Group Resource Limits is not present.
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to retrieve the Network Group
   *           Resource Limits because the client does not have the
   *           correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  NetworkGroupResourceLimitsCfgClient getNetworkGroupResourceLimits()
      throws DefinitionDecodingException, ManagedObjectDecodingException,
      ManagedObjectNotFoundException, ConcurrentModificationException,
      AuthorizationException, CommunicationException;



  /**
   * Creates a new Network Group Resource Limits. The new Network
   * Group Resource Limits will initially not contain any property
   * values (including mandatory properties). Once the Network Group
   * Resource Limits has been configured it can be added to the server
   * using the {@link #commit()} method.
   *
   * @param <C>
   *          The type of the Network Group Resource Limits being
   *          created.
   * @param d
   *          The definition of the Network Group Resource Limits to
   *          be created.
   * @param exceptions
   *          An optional collection in which to place any {@link
   *          DefaultBehaviorException}s that occurred whilst
   *          attempting to determine the default values of the Network
   *          Group Resource Limits. This argument can be
   *          <code>null<code>.
   * @return Returns a new Network Group Resource Limits configuration
   *         instance.
   */
  <C extends NetworkGroupResourceLimitsCfgClient> C createNetworkGroupResourceLimits(
      ManagedObjectDefinition<C, ? extends NetworkGroupResourceLimitsCfg> d, Collection<DefaultBehaviorException> exceptions);



  /**
   * Removes the Network Group Resource Limits if it exists.
   *
   * @throws ManagedObjectNotFoundException
   *           If the Network Group Resource Limits does not exist.
   * @throws OperationRejectedException
   *           If the server refuses to remove the Network Group
   *           Resource Limits due to some server-side constraint which
   *           cannot be satisfied (for example, if it is referenced by
   *           another managed object).
   * @throws ConcurrentModificationException
   *           If this Network Group has been removed from the server
   *           by another client.
   * @throws AuthorizationException
   *           If the server refuses to remove the Network Group
   *           Resource Limits because the client does not have the
   *           correct privileges.
   * @throws CommunicationException
   *           If the client cannot contact the server due to an
   *           underlying communication problem.
   */
  void removeNetworkGroupResourceLimits()
      throws ManagedObjectNotFoundException, OperationRejectedException,
      ConcurrentModificationException, AuthorizationException,
      CommunicationException;

}
