/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.server;



import java.util.SortedSet;
import org.opends.server.admin.Configuration;
import org.opends.server.admin.server.ConfigurationAddListener;
import org.opends.server.admin.server.ConfigurationChangeListener;
import org.opends.server.admin.server.ConfigurationDeleteListener;
import org.opends.server.config.ConfigException;
import org.opends.server.types.DN;



/**
 * A server-side interface for querying Network Group settings.
 * <p>
 * The Network Group is used to classify incoming connections and
 * route requests to workflows.
 */
public interface NetworkGroupCfg extends Configuration {

  /**
   * Gets the configuration class associated with this Network Group.
   *
   * @return Returns the configuration class associated with this Network Group.
   */
  Class<? extends NetworkGroupCfg> configurationClass();



  /**
   * Register to be notified when this Network Group is changed.
   *
   * @param listener
   *          The Network Group configuration change listener.
   */
  void addChangeListener(ConfigurationChangeListener<NetworkGroupCfg> listener);



  /**
   * Deregister an existing Network Group configuration change listener.
   *
   * @param listener
   *          The Network Group configuration change listener.
   */
  void removeChangeListener(ConfigurationChangeListener<NetworkGroupCfg> listener);



  /**
   * Gets the "enabled" property.
   * <p>
   * Indicates whether the Network Group is enabled for use in the
   * server.
   * <p>
   * If a network group is not enabled, its workflows will not be
   * accessible when processing operations.
   *
   * @return Returns the value of the "enabled" property.
   */
  boolean isEnabled();



  /**
   * Gets the "network-group-id" property.
   * <p>
   * Specifies the name that is used to identify the associated
   * Network Group .
   * <p>
   * The name must be unique among all the Network Groups in the
   * server.
   *
   * @return Returns the value of the "network-group-id" property.
   */
  String getNetworkGroupId();



  /**
   * Gets the "priority" property.
   * <p>
   * Specifies the order in which the network groups are evaluated.
   * <p>
   * A client connection is first compared against network group with
   * priority 1. If the client connection does not match the network
   * group criteria, the client connection is compared against network
   * group with priority 2 etc...
   *
   * @return Returns the value of the "priority" property.
   */
  int getPriority();



  /**
   * Gets the "workflow" property.
   * <p>
   * Identifies the workflows in the network group.
   *
   * @return Returns an unmodifiable set containing the values of the "workflow" property.
   */
  SortedSet<String> getWorkflow();



  /**
   * Gets the "workflow" property as a set of DNs.
   * <p>
   * Identifies the workflows in the network group.
   *
   * @return Returns the DN values of the "workflow" property.
   */
  SortedSet<DN> getWorkflowDNs();



  /**
   * Determines whether or not the Network Group Criteria exists.
   *
   * @return Returns <true> if the Network Group Criteria exists.
   */
  boolean hasNetworkGroupCriteria();



  /**
   * Gets the Network Group Criteria if it is present.
   *
   * @return Returns the Network Group Criteria if it is present.
   * @throws ConfigException
   *           If the Network Group Criteria does not exist or it could not
   *           be successfully decoded.
   */
  NetworkGroupCriteriaCfg getNetworkGroupCriteria() throws ConfigException;



  /**
   * Registers to be notified when the Network Group Criteria is added.
   *
   * @param listener
   *          The Network Group Criteria configuration add listener.
   * @throws ConfigException
   *          If the add listener could not be registered.
   */
  void addNetworkGroupCriteriaAddListener(ConfigurationAddListener<NetworkGroupCriteriaCfg> listener) throws ConfigException;



  /**
   * Deregisters an existing Network Group Criteria configuration add listener.
   *
   * @param listener
   *          The Network Group Criteria configuration add listener.
   */
  void removeNetworkGroupCriteriaAddListener(ConfigurationAddListener<NetworkGroupCriteriaCfg> listener);



  /**
   * Registers to be notified the Network Group Criteria is deleted.
   *
   * @param listener
   *          The Network Group Criteria configuration delete listener.
   * @throws ConfigException
   *          If the delete listener could not be registered.
   */
  void addNetworkGroupCriteriaDeleteListener(ConfigurationDeleteListener<NetworkGroupCriteriaCfg> listener) throws ConfigException;



  /**
   * Deregisters an existing Network Group Criteria configuration delete listener.
   *
   * @param listener
   *          The Network Group Criteria configuration delete listener.
   */
  void removeNetworkGroupCriteriaDeleteListener(ConfigurationDeleteListener<NetworkGroupCriteriaCfg> listener);



  /**
   * Determines whether or not the Network Group Request Filtering Policy exists.
   *
   * @return Returns <true> if the Network Group Request Filtering Policy exists.
   */
  boolean hasNetworkGroupRequestFilteringPolicy();



  /**
   * Gets the Network Group Request Filtering Policy if it is present.
   *
   * @return Returns the Network Group Request Filtering Policy if it is present.
   * @throws ConfigException
   *           If the Network Group Request Filtering Policy does not exist or it could not
   *           be successfully decoded.
   */
  NetworkGroupRequestFilteringPolicyCfg getNetworkGroupRequestFilteringPolicy() throws ConfigException;



  /**
   * Registers to be notified when the Network Group Request Filtering Policy is added.
   *
   * @param listener
   *          The Network Group Request Filtering Policy configuration add listener.
   * @throws ConfigException
   *          If the add listener could not be registered.
   */
  void addNetworkGroupRequestFilteringPolicyAddListener(ConfigurationAddListener<NetworkGroupRequestFilteringPolicyCfg> listener) throws ConfigException;



  /**
   * Deregisters an existing Network Group Request Filtering Policy configuration add listener.
   *
   * @param listener
   *          The Network Group Request Filtering Policy configuration add listener.
   */
  void removeNetworkGroupRequestFilteringPolicyAddListener(ConfigurationAddListener<NetworkGroupRequestFilteringPolicyCfg> listener);



  /**
   * Registers to be notified the Network Group Request Filtering Policy is deleted.
   *
   * @param listener
   *          The Network Group Request Filtering Policy configuration delete listener.
   * @throws ConfigException
   *          If the delete listener could not be registered.
   */
  void addNetworkGroupRequestFilteringPolicyDeleteListener(ConfigurationDeleteListener<NetworkGroupRequestFilteringPolicyCfg> listener) throws ConfigException;



  /**
   * Deregisters an existing Network Group Request Filtering Policy configuration delete listener.
   *
   * @param listener
   *          The Network Group Request Filtering Policy configuration delete listener.
   */
  void removeNetworkGroupRequestFilteringPolicyDeleteListener(ConfigurationDeleteListener<NetworkGroupRequestFilteringPolicyCfg> listener);



  /**
   * Determines whether or not the Network Group Resource Limits exists.
   *
   * @return Returns <true> if the Network Group Resource Limits exists.
   */
  boolean hasNetworkGroupResourceLimits();



  /**
   * Gets the Network Group Resource Limits if it is present.
   *
   * @return Returns the Network Group Resource Limits if it is present.
   * @throws ConfigException
   *           If the Network Group Resource Limits does not exist or it could not
   *           be successfully decoded.
   */
  NetworkGroupResourceLimitsCfg getNetworkGroupResourceLimits() throws ConfigException;



  /**
   * Registers to be notified when the Network Group Resource Limits is added.
   *
   * @param listener
   *          The Network Group Resource Limits configuration add listener.
   * @throws ConfigException
   *          If the add listener could not be registered.
   */
  void addNetworkGroupResourceLimitsAddListener(ConfigurationAddListener<NetworkGroupResourceLimitsCfg> listener) throws ConfigException;



  /**
   * Deregisters an existing Network Group Resource Limits configuration add listener.
   *
   * @param listener
   *          The Network Group Resource Limits configuration add listener.
   */
  void removeNetworkGroupResourceLimitsAddListener(ConfigurationAddListener<NetworkGroupResourceLimitsCfg> listener);



  /**
   * Registers to be notified the Network Group Resource Limits is deleted.
   *
   * @param listener
   *          The Network Group Resource Limits configuration delete listener.
   * @throws ConfigException
   *          If the delete listener could not be registered.
   */
  void addNetworkGroupResourceLimitsDeleteListener(ConfigurationDeleteListener<NetworkGroupResourceLimitsCfg> listener) throws ConfigException;



  /**
   * Deregisters an existing Network Group Resource Limits configuration delete listener.
   *
   * @param listener
   *          The Network Group Resource Limits configuration delete listener.
   */
  void removeNetworkGroupResourceLimitsDeleteListener(ConfigurationDeleteListener<NetworkGroupResourceLimitsCfg> listener);

}
