/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2009-2010 Sun Microsystems, Inc.
 */


package org.opends.server.backends.jeb;

import org.opends.server.api.ExtensibleIndexer;
import org.opends.server.types.AttributeType;
import org.opends.server.types.AttributeValue;
import org.opends.server.types.Entry;
import org.opends.server.types.Modification;
import org.opends.server.api.ExtensibleMatchingRule;
import org.opends.server.types.Attribute;


import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;


/**
 *This class implements an Indexer for extensible matching rules in JE Backend.
 */
public final class JEExtensibleIndexer extends Indexer
{
  /**
   * The comparator for index keys generated by this class.
   */
  private static final Comparator<byte[]> comparator =
       new AttributeIndex.KeyComparator();



  /**
   * The attribute type for which this instance will
   * generate index keys.
   */
  private final AttributeType attributeType;



  /**
   * The extensible indexer which will generate the keys
   * for the associated  extensible matching rule.
   */
  private final ExtensibleIndexer extensibleIndexer;



  /**
   * Creates a new extensible indexer for JE backend.
   *
   * @param attributeType The attribute type for which an indexer is
   *                                            required.
   * @param matchingRule  The extensible matching rule to be indexed.
   * @param extensibleIndexer The extensible indexer to be used.
   */
  public JEExtensibleIndexer(AttributeType attributeType,
          ExtensibleMatchingRule matchingRule,
          ExtensibleIndexer extensibleIndexer)
  {
    this.attributeType = attributeType;
    this.extensibleIndexer = extensibleIndexer;
  }



   /**
   * Gets a string representation of this object.  The returned value is
   * used to name an index created using this object.
   * @return A string representation of this object.
   */
  @Override
  public String toString()
  {
    return attributeType.getNameOrOID() + "."
            + extensibleIndexer.getExtensibleIndexID();
  }



  /**
   * Gets the comparator that must be used to compare index keys
   * generated by this class.
   *
   * @return A byte array comparator.
   */
  @Override
  public Comparator<byte[]> getComparator()
  {
    return comparator;
  }



  /**
   * {@inheritDoc}
   */
  @Override
  public void indexEntry(Entry entry, Set<byte[]> keys)
  {
    List<Attribute> attrList =
         entry.getAttribute(attributeType);
    if (attrList != null)
    {
      indexAttribute(attrList, keys);
    }
  }



 /**
   * {@inheritDoc}
   */
  @Override
  public void replaceEntry(Entry oldEntry, Entry newEntry,
                           Map<byte[], Boolean> modifiedKeys)
  {
    List<Attribute> newAttributes = newEntry.getAttribute(attributeType, true);
    List<Attribute> oldAttributes = oldEntry.getAttribute(attributeType, true);

    indexAttribute(oldAttributes, modifiedKeys, false);
    indexAttribute(newAttributes, modifiedKeys, true);
  }




  /**
   * {@inheritDoc}
   */
  @Override
  public void modifyEntry(Entry oldEntry, Entry newEntry,
                          List<Modification> mods,
                          Map<byte[], Boolean> modifiedKeys)
  {
    List<Attribute> newAttributes = newEntry.getAttribute(attributeType, true);
    List<Attribute> oldAttributes = oldEntry.getAttribute(attributeType, true);

    indexAttribute(oldAttributes, modifiedKeys, false);
    indexAttribute(newAttributes, modifiedKeys, true);
  }



  /**
   * Generates the set of extensible  index keys for an attribute.
   * @param attrList The attribute for which substring keys are required.
   * @param keys The set into which the generated keys will be inserted.
   */
  private void indexAttribute(List<Attribute> attrList,
                              Set<byte[]> keys)
  {
    if (attrList == null) return;

    for (Attribute attr : attrList)
    {
      if (attr.isVirtual())
      {
        continue;
      }
      for (AttributeValue value : attr)
      {
        extensibleIndexer.getKeys(value, keys);
      }
    }
  }



  /**
   * Generates the set of index keys for an attribute.
   * @param attrList The attribute to be indexed.
   * @param modifiedKeys The map into which the modified
   * keys will be inserted.
   * @param insert <code>true</code> if generated keys should
   * be inserted or <code>false</code> otherwise.
   */
  private void indexAttribute(List<Attribute> attrList,
                              Map<byte[], Boolean> modifiedKeys,
                              Boolean insert)
  {
    if (attrList == null) return;

    for (Attribute attr : attrList)
    {
      if (attr.isVirtual())
      {
        continue;
      }
      for (AttributeValue value : attr)
      {
        extensibleIndexer.getKeys(value,modifiedKeys,insert);
      }
    }
  }
}
