/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE
 * or https://OpenDS.dev.java.net/OpenDS.LICENSE.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at
 * trunk/opends/resource/legal-notices/OpenDS.LICENSE.  If applicable,
 * add the following below this CDDL HEADER, with the fields enclosed
 * by brackets "[]" replaced with your own identifying information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 */
package org.opends.server.admin.std.server;



import java.util.SortedSet;
import org.opends.server.admin.server.ConfigurationChangeListener;
import org.opends.server.types.DN;



/**
 * A server-side interface for querying File Based Access Log
 * Publisher settings.
 * <p>
 * File Based Access Log Publishers publish access messages to the
 * file system.
 */
public interface FileBasedAccessLogPublisherCfg extends AccessLogPublisherCfg {

  /**
   * Gets the configuration class associated with this File Based Access Log Publisher.
   *
   * @return Returns the configuration class associated with this File Based Access Log Publisher.
   */
  Class<? extends FileBasedAccessLogPublisherCfg> configurationClass();



  /**
   * Register to be notified when this File Based Access Log Publisher is changed.
   *
   * @param listener
   *          The File Based Access Log Publisher configuration change listener.
   */
  void addFileBasedAccessChangeListener(ConfigurationChangeListener<FileBasedAccessLogPublisherCfg> listener);



  /**
   * Deregister an existing File Based Access Log Publisher configuration change listener.
   *
   * @param listener
   *          The File Based Access Log Publisher configuration change listener.
   */
  void removeFileBasedAccessChangeListener(ConfigurationChangeListener<FileBasedAccessLogPublisherCfg> listener);



  /**
   * Gets the "append" property.
   * <p>
   * Specifies whether to append to existing log files.
   *
   * @return Returns the value of the "append" property.
   */
  boolean isAppend();



  /**
   * Gets the "asynchronous" property.
   * <p>
   * Indicates whether the File Based Access Log Publisher will
   * publish records asynchronously.
   *
   * @return Returns the value of the "asynchronous" property.
   */
  boolean isAsynchronous();



  /**
   * Gets the "auto-flush" property.
   * <p>
   * Specifies whether to flush the writer after every log record.
   * <p>
   * If the asynchronous writes option is used, the writer is flushed
   * after all the log records in the queue are written.
   *
   * @return Returns the value of the "auto-flush" property.
   */
  boolean isAutoFlush();



  /**
   * Gets the "buffer-size" property.
   * <p>
   * Specifies the log file buffer size.
   *
   * @return Returns the value of the "buffer-size" property.
   */
  long getBufferSize();



  /**
   * Gets the "java-class" property.
   * <p>
   * The fully-qualified name of the Java class that provides the File
   * Based Access Log Publisher implementation.
   *
   * @return Returns the value of the "java-class" property.
   */
  String getJavaClass();



  /**
   * Gets the "log-file" property.
   * <p>
   * The file name to use for the log files generated by the File
   * Based Access Log Publisher. The path to the file is relative to
   * the server root.
   *
   * @return Returns the value of the "log-file" property.
   */
  String getLogFile();



  /**
   * Gets the "log-file-permissions" property.
   * <p>
   * The UNIX permissions of the log files created by this File Based
   * Access Log Publisher.
   *
   * @return Returns the value of the "log-file-permissions" property.
   */
  String getLogFilePermissions();



  /**
   * Gets the "queue-size" property.
   * <p>
   * The maximum number of log records that can be stored in the
   * asynchronous queue.
   * <p>
   * Setting the queue size to zero activates parallel log writer
   * implementation which has no queue size limit and as such the
   * parallel log writer should only be used on a very well tuned
   * server configuration to avoid potential out of memory errors.
   *
   * @return Returns the value of the "queue-size" property.
   */
  int getQueueSize();



  /**
   * Gets the "retention-policy" property.
   * <p>
   * The retention policy to use for the File Based Access Log
   * Publisher .
   * <p>
   * When multiple policies are used, log files are cleaned when any
   * of the policy's conditions are met.
   *
   * @return Returns an unmodifiable set containing the values of the "retention-policy" property.
   */
  SortedSet<String> getRetentionPolicy();



  /**
   * Gets the "retention-policy" property as a set of DNs.
   * <p>
   * The retention policy to use for the File Based Access Log
   * Publisher .
   * <p>
   * When multiple policies are used, log files are cleaned when any
   * of the policy's conditions are met.
   *
   * @return Returns the DN values of the "retention-policy" property.
   */
  SortedSet<DN> getRetentionPolicyDNs();



  /**
   * Gets the "rotation-policy" property.
   * <p>
   * The rotation policy to use for the File Based Access Log
   * Publisher .
   * <p>
   * When multiple policies are used, rotation will occur if any
   * policy's conditions are met.
   *
   * @return Returns an unmodifiable set containing the values of the "rotation-policy" property.
   */
  SortedSet<String> getRotationPolicy();



  /**
   * Gets the "rotation-policy" property as a set of DNs.
   * <p>
   * The rotation policy to use for the File Based Access Log
   * Publisher .
   * <p>
   * When multiple policies are used, rotation will occur if any
   * policy's conditions are met.
   *
   * @return Returns the DN values of the "rotation-policy" property.
   */
  SortedSet<DN> getRotationPolicyDNs();



  /**
   * Gets the "time-interval" property.
   * <p>
   * Specifies the interval at which to check whether the log files
   * need to be rotated.
   *
   * @return Returns the value of the "time-interval" property.
   */
  long getTimeInterval();

}
