<?php
/**
 * TEditCommandColumn class file
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @link http://www.pradosoft.com/
 * @copyright Copyright &copy; 2005 PradoSoft
 * @license http://www.pradosoft.com/license/
 * @version $Revision: $  $Date: $
 * @package System.Web.UI.WebControls
 */

/**
 * TDataGridColumn class file
 */
Prado::using('System.Web.UI.WebControls.TDataGridColumn');

/**
 * TEditCommandColumn class
 *
 * TEditCommandColumn contains the Edit command buttons for editing data items in each row.
 *
 * TEditCommandColumn will create an edit button if a cell is not in edit mode.
 * Otherwise an update button and a cancel button will be created within the cell.
 * The button captions are specified using {@link setEditText EditText},
 * {@link setUpdateText UpdateText}, and {@link setCancelText CancelText}.
 *
 * The buttons in the column can be set to display as hyperlinks or push buttons
 * by setting the {@link setButtonType ButtonType} property.
 *
 * When an edit button is clicked, the datagrid will generate an
 * {@link onEditCommand OnEditCommand} event. When an update/cancel button
 * is clicked, the datagrid will generate an
 * {@link onUpdateCommand OnUpdateCommand} or an {@link onCancelCommand OnCancelCommand}
 * You can write these event handlers to change the state of specific datagrid item.
 *
 * The {@link setCausesValidation CausesValidation} and {@link setValidationGroup ValidationGroup}
 * properties affect the corresponding properties of the edit and update buttons.
 * The cancel button does not cause validation by default.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @version $Revision: $  $Date: $
 * @package System.Web.UI.WebControls
 * @since 3.0
 */
class TEditCommandColumn extends TDataGridColumn
{
	/**
	 * @return string the type of command button. Defaults to LinkButton.
	 */
	public function getButtonType()
	{
		return $this->getViewState('ButtonType','LinkButton');
	}

	/**
	 * @param string the type of command button, LinkButton or PushButton
	 */
	public function setButtonType($value)
	{
		$this->setViewState('ButtonType',TPropertyValue::ensureEnum($value,'LinkButton','PushButton'),'LinkButton');
	}

	/**
	 * @return string the caption of the edit button. Defaults to 'Edit'.
	 */
	public function getEditText()
	{
		return $this->getViewState('EditText','Edit');
	}

	/**
	 * @param string the caption of the edit button
	 */
	public function setEditText($value)
	{
		$this->setViewState('EditText',$value,'Edit');
	}

	/**
	 * @return string the caption of the update button. Defaults to 'Update'.
	 */
	public function getUpdateText()
	{
		return $this->getViewState('UpdateText','Update');
	}

	/**
	 * @param string the caption of the update button
	 */
	public function setUpdateText($value)
	{
		$this->setViewState('UpdateText',$value,'Update');
	}

	/**
	 * @return string the caption of the cancel button. Defaults to 'Cancel'.
	 */
	public function getCancelText()
	{
		return $this->getViewState('CancelText','Cancel');
	}

	/**
	 * @param string the caption of the cancel button
	 */
	public function setCancelText($value)
	{
		$this->setViewState('CancelText',$value,'Cancel');
	}

	/**
	 * @return boolean whether postback event trigger by edit or update button will cause input validation, default is true
	 */
	public function getCausesValidation()
	{
		return $this->getViewState('CausesValidation',true);
	}

	/**
	 * @param boolean whether postback event trigger by edit or update button will cause input validation
	 */
	public function setCausesValidation($value)
	{
		$this->setViewState('CausesValidation',TPropertyValue::ensureBoolean($value),true);
	}

	/**
	 * @return string the group of validators which the edit or update button causes validation upon postback
	 */
	public function getValidationGroup()
	{
		return $this->getViewState('ValidationGroup','');
	}

	/**
	 * @param string the group of validators which the edit or update button causes validation upon postback
	 */
	public function setValidationGroup($value)
	{
		$this->setViewState('ValidationGroup',$value,'');
	}

	/**
	 * Initializes the specified cell to its initial values.
	 * This method overrides the parent implementation.
	 * It creates an update and a cancel button for cell in edit mode.
	 * Otherwise it creates an edit button.
	 * @param TTableCell the cell to be initialized.
	 * @param integer the index to the Columns property that the cell resides in.
	 * @param string the type of cell (Header,Footer,Item,AlternatingItem,EditItem,SelectedItem)
	 */
	public function initializeCell($cell,$columnIndex,$itemType)
	{
		parent::initializeCell($cell,$columnIndex,$itemType);
		if($itemType===TDataGrid::IT_ITEM || $itemType===TDataGrid::IT_ALTERNATINGITEM || $itemType===TDataGrid::IT_SELECTEDITEM)
			$cell->getControls()->add($this->createButton('Edit',$this->getEditText(),false,''));
		else if($itemType===TDataGrid::IT_EDITITEM)
		{
			$controls=$cell->getControls();
			$controls->add($this->createButton('Update',$this->getUpdateText(),$this->getCausesValidation(),$this->getValidationGroup()));
			$controls->add('&nbsp;');
			$controls->add($this->createButton('Cancel',$this->getCancelText(),false,''));
		}
	}

	/**
	 * Creates a button and initializes its properties.
	 * The button type is determined by {@link getButtonType ButtonType}.
	 * @param string command name associated with the button
	 * @param string button caption
	 * @param boolean whether the button should cause validation
	 * @param string the validation group that the button belongs to
	 * @return mixed the newly created button.
	 */
	protected function createButton($commandName,$text,$causesValidation,$validationGroup)
	{
		if($this->getButtonType()==='LinkButton')
			$button=Prado::createComponent('System.Web.UI.WebControls.TLinkButton');
		else
			$button=Prado::createComponent('System.Web.UI.WebControls.TButton');
		$button->setText($text);
		$button->setCommandName($commandName);
		$button->setCausesValidation($causesValidation);
		$button->setValidationGroup($validationGroup);
		return $button;
	}
}

?>