/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT 
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the 
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.app.vo;

import com.google.gwt.user.client.rpc.IsSerializable;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

/**
 * Form layout describes metadata for controls layout.
 * This class encapsulates all info about controls position, colspan/rowspan and width.
 * Typically its structure looks like this:<br/>
 * <src>
 * [header][client width for col#0][client width for col#1]...[/header]<br/>
 * [row#0][col#0 fieldID="field00" rowspan="1" colspan="1"][col#1 fieldID="field01" rowspan="1" colspan="1"]...[/row#0]<br/>
 * [row#1][col#0 fieldID="field10" rowspan="1" colspan="1"][col#1 fieldID="field11" rowspan="1" colspan="1"]...[/row#1]<br/>
 * </src>
 * @author Sultan Tezadov, Dmitry Antonov
 * @since 8 Dec 2006
 */
public class FormLayoutMeta implements IsSerializable {
    
    public final static int CLIENT_WIDTH_NOT_SPECIFIED = 0;
    
    /**
     * Stores client width for all controls which are placed in this column.
     * If <src>clientWidth</src> field is set - all input fields in this column shall be shrink/enlarge
     * to required value.<br/><br/>
     * <i><b>Attention:</b> if clientWidth equals to <src>CLIENT_WIDTH_NOT_SPECIFIED</src> then all controls shall
     * retain their own size.</i>
     */
    public static class ColumnHeader implements IsSerializable {
        private int clientWidth;
        
        public ColumnHeader(int clientWidth) {
            setClientWidth(clientWidth);
        }
        
        public ColumnHeader() {
            this(CLIENT_WIDTH_NOT_SPECIFIED);
        }

        public int getClientWidth () {
            return clientWidth;
        }

        public void setClientWidth (int clientWidth) {
            this.clientWidth = clientWidth;
        }
        
    }
    
    /**
     * Encapsulates all data about form row. Actually this is only container for Col objects yet.
     */
    public static class Row implements IsSerializable {
        private Col[] cols;
        public Row() {
            this(new Col[0]);
        }
        public Row(Col[] cols) {
            setCols(cols);
        }
        public void setCols(Col[] cols) {
            this.cols = cols;
        }
        public int getColsCount() {
            return getCols().length;
        }

        public Col[] getCols () {
            return cols;
        }
    }

    /**
     * Primary class for layout metadata. Stores information about field which is placed in
     * this column: name, colspan and rowspan.<br/>
     * Note: doesn't support empty fields yet. Should be added later.
     */
    public static class Col implements IsSerializable {
        private String fieldID = "";
        private int colspan = 1;
        private int rowspan = 1;
        public void setFieldID(String fieldID) {
            this.fieldID = fieldID;
        }

        public String getFieldID () {
            return fieldID;
        }

        public int getColspan () {
            return colspan;
        }

        public void setColspan (int colspan) {
            this.colspan = colspan < 1 ? 1 : colspan;
        }

        public int getRowspan () {
            return rowspan;
        }

        public void setRowspan (int rowspan) {
            this.rowspan = rowspan < 1 ? 1 : rowspan;
        }

        public Col() {
            this("");
        }
        public Col(String fieldID) {
            this(fieldID, 1, 1);
        }
        public Col(String fieldID, int rowspan, int colspan) {
            setFieldID (fieldID);
            setRowspan(rowspan);
            setColspan(colspan);
        }
    }
    
    private Row[] rows;
    private ColumnHeader[] headers;
    private String[] hiddenControls;
    
    
    public FormLayoutMeta() {
        this(new Row[0], null, new String[0]);
    }
    
    public FormLayoutMeta(Row[] rows) {
        this(rows, null, new String[0]);
    }
    
    public FormLayoutMeta(Row[] rows, ColumnHeader[] headers, String[] hiddenControls) {
        setRows(rows);
        setHeaders(headers);
        setHiddenControls(hiddenControls);
    }

    /**
     * Calculates count of columns in current layout.
     * In fact this is just a max count of a cols.
     * @return width of the layout.
     */
    public int getWidth() {
        int result = 0;
        for (int i = 0; i < getRows().length; i++) {
            if (result < getRows()[i].getColsCount ()) {
                result = getRows()[i].getColsCount ();
            }
        }
        return result;
    }
    
    public Row[] getRows () {
        return rows;
    }

    public void setRows (Row[] rows) {
        this.rows = rows;
    }

    public ColumnHeader[] getHeaders () {
        return headers;
    }
    
    public String[] getHiddenControls() {
        return hiddenControls;
    }
    
    public void setHiddenControls(String[] hiddenControls) {
        this.hiddenControls = hiddenControls;
    }

    public void setHeaders (ColumnHeader[] headers) {
        this.headers = headers;
    }


    public ColumnHeader getHeader(int index) {
        if (headers == null || index >= headers.length) {
            return null;
        }
        else {
            return headers[index];
        }
    }
    
    public boolean isControlHidden(String fieldId) {
        for (int i = 0; i < hiddenControls.length; i++) {
            String hiddenControl = hiddenControls[i];
            if (hiddenControl.hashCode() == fieldId.hashCode() && 
                hiddenControl.equals(fieldId)) {
                return true;
            }
        }
        return false;
    }
    
    public boolean containHeaders() {
        return headers != null && headers.length > 0;
    }

}
