/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.config.utils;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.modules.config.jxb.Config;
import com.queplix.core.utils.cache.Cache;
import com.queplix.core.utils.cache.CacheFactory;
import com.queplix.core.utils.dao.AbstractPropertyFactory;

/**
 * Config system property access factory
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.2 $ $Date: 2005/10/26 14:03:51 $
 */

public final class ConfigPropertyFactory
    extends AbstractPropertyFactory {

    // ================================================================= Constants

    // Single instance.
    private static final ConfigPropertyFactory o = new ConfigPropertyFactory();

    // Config file name.
    private static final String CONFIG_FILE = "config.xml";

    // ================================================================= Fields

    // XML config
    private Config config;

    // DAO cached clasess.
    private EntityViewConfigDAO entityViewConfigDAO;
    private ViewObjectsDAO viewObjectsDAO;
//    private CaptionConfigDAO captionConfigDAO;
    private CustomConfigDAO customConfigDAO;
    private UserPropertyConfigDAO userPropertyConfigDAO;
    private SysPropertyDAO sysPropertyDAO;
    private ContextMenuConfigDAO contextMenuConfigDAO;
    private LocalizationDAO localizationDAO;
    
    // Cache cached classes.
    private Cache entityViewConfigCache;
    private Cache viewObjectsCache;
//    private Cache captionConfigCache;
    private Cache customConfigCache;
    private Cache userPropertyConfigCache;
    private Cache sysPropertyCache;
    private Cache contextMenuConfigCache;
    private Cache localizationCache;

    // ================================================================= Constructor

    //
    // Constructor - blocks direct instantiation.
    //
    private ConfigPropertyFactory() {

        // XML config initialization
        config = ( Config ) loadSysPropertiesAsObject( CONFIG_FILE, Config.class );

        //
        // DAO initialization
        //

        // entity view config
        entityViewConfigDAO = ( EntityViewConfigDAO ) initObject(
            config.getEntityviewConfig().getDao().getClassName(),
            config.getEntityviewConfig().getDao().getParam() );

        // viewObjects config
        viewObjectsDAO = (ViewObjectsDAO) initObject(
            config.getViewObjects().getDao().getClassName(),
            config.getViewObjects().getDao().getParam() );

        // custom config
        customConfigDAO = ( CustomConfigDAO ) initObject(
            config.getCustomConfig().getDao().getClassName(),
            config.getCustomConfig().getDao().getParam() );

        // user property config
        userPropertyConfigDAO = ( UserPropertyConfigDAO ) initObject(
            config.getUserpropertyConfig().getDao().getClassName(),
            config.getUserpropertyConfig().getDao().getParam() );

        // system property config
        sysPropertyDAO = ( SysPropertyDAO ) initObject(
            config.getSyspropertyConfig().getDao().getClassName(),
            config.getSyspropertyConfig().getDao().getParam() );

        // conetxt menu config
        contextMenuConfigDAO = ( ContextMenuConfigDAO ) initObject(
            config.getContextMenuConfig().getDao().getClassName(),
            config.getContextMenuConfig().getDao().getParam() );

        // conetxt menu config
        localizationDAO = (LocalizationDAO) initObject(
            config.getLocalizationConfig().getDao().getClassName(),
            config.getLocalizationConfig().getDao().getParam() );

        //
        // Cache initialization
        //

        // entity view cache
        entityViewConfigCache = ( Cache ) CacheFactory.getInstance().initObject(
            config.getEntityviewConfig().getCache().getClassName(),
            config.getEntityviewConfig().getCache().getParam() );

        // focus cache
        viewObjectsCache = ( Cache ) CacheFactory.getInstance().initObject(
            config.getViewObjects().getCache().getClassName(),
            config.getViewObjects().getCache().getParam() );

        // custom cache
        customConfigCache = ( Cache ) CacheFactory.getInstance().initObject(
            config.getCustomConfig().getCache().getClassName(),
            config.getCustomConfig().getCache().getParam() );

        // user property cache
        userPropertyConfigCache = ( Cache ) CacheFactory.getInstance().initObject(
            config.getUserpropertyConfig().getCache().getClassName(),
            config.getUserpropertyConfig().getCache().getParam() );

        // system property cache
        sysPropertyCache = ( Cache ) CacheFactory.getInstance().initObject(
            config.getSyspropertyConfig().getCache().getClassName(),
            config.getSyspropertyConfig().getCache().getParam() );

        // context menu cache
        contextMenuConfigCache = ( Cache ) CacheFactory.getInstance().initObject(
            config.getContextMenuConfig().getCache().getClassName(),
            config.getContextMenuConfig().getCache().getParam() );
        // context menu cache
        localizationCache = (Cache) CacheFactory.getInstance().initObject(
            config.getLocalizationConfig().getCache().getClassName(),
            config.getLocalizationConfig().getCache().getParam() );
    }

    // ================================================================= Static methods

    /**
     * Get single instance
     * @return ConfigPropertyFactory object
     */
    public static ConfigPropertyFactory getInstance() {
        return o;
    }

    // ================================================================= Public methods

    /**
     * Instantiates the entity view config DAO
     * @return DAO
     */
    public EntityViewConfigDAO getEntityViewConfigDAO() {
        if( entityViewConfigDAO == null ) {
            throw new GenericSystemException( "Can't get entity view config DAO" );
        }
        return entityViewConfigDAO;
    }

    /**
     * Instantiates the entity view config cache
     * @return cache
     */
    public Cache getEntityViewConfigCache() {
        if( entityViewConfigCache == null ) {
            throw new GenericSystemException( "Can't get entity view config cache" );
        }
        return entityViewConfigCache;
    }

    /**
     * Instantiates the view objects DAO
     * @return DAO
     */
    public ViewObjectsDAO getViewObjectsDAO() {
        if( viewObjectsDAO == null ) {
            throw new GenericSystemException( "Can't get view objects DAO" );
        }
        return viewObjectsDAO;
    }

    /**
     * Instantiates the focus config cache
     * @return cache
     */
    public Cache getViewObjectsCache() {
        if( viewObjectsCache == null ) {
            throw new GenericSystemException( "Can't get focus config cache" );
        }
        return viewObjectsCache;
    }

    /**
    * Instantiates the caption config DAO
    * @return DAO
    */
    public LocalizationDAO getLocalizationDAO() {
        if (localizationDAO == null) {
            throw new GenericSystemException("Can't get localization DAO");
        }
        return localizationDAO;
    }

    /**
     * Instantiates the caption config cache
     * @return cache
     */
    public Cache getLocalizationCache() {
        if(localizationCache == null) {
            throw new GenericSystemException("Can't get localization cache");
        }
        return localizationCache;
    }

    /**
     * Instantiates the custom config DAO
     * @return DAO
     */
    public CustomConfigDAO getCustomConfigDAO() {
        if( customConfigDAO == null ) {
            throw new GenericSystemException( "Can't get custom config DAO" );
        }
        return customConfigDAO;
    }

    /**
     * Instantiates the custom config cache
     * @return cache
     */
    public Cache getCustomConfigCache() {
        if( customConfigCache == null ) {
            throw new GenericSystemException( "Can't get custom config cache" );
        }
        return customConfigCache;
    }

    /**
     * Instantiates the user property config DAO
     * @return DAO
     */
    public UserPropertyConfigDAO getUserPropertyConfigDAO() {
        if( userPropertyConfigDAO == null ) {
            throw new GenericSystemException( "Can't get user property config DAO" );
        }
        return userPropertyConfigDAO;
    }

    /**
     * Instantiates the user property config cache
     * @return cache
     */
    public Cache getUserPropertyConfigCache() {
        if( userPropertyConfigCache == null ) {
            throw new GenericSystemException( "Can't get user property config cache" );
        }
        return userPropertyConfigCache;
    }

    /**
     * Instantiates the system property DAO
     * @return DAO
     */
    public SysPropertyDAO getSysPropertyDAO() {
        if( sysPropertyDAO == null ) {
            throw new GenericSystemException( "Can't get system property DAO" );
        }
        return sysPropertyDAO;
    }

    /**
     * Instantiates the system property cache
     * @return cache
     */
    public Cache getSysPropertyCache() {
        if( sysPropertyCache == null ) {
            throw new GenericSystemException( "Can't get system property cache" );
        }
        return sysPropertyCache;
    }

    /**
     * Instantiates the context menu config DAO
     * @return DAO
     */
    public ContextMenuConfigDAO getContextMenuConfigDAO() {
        if( contextMenuConfigDAO == null ) {
            throw new GenericSystemException( "Can't get context menu config DAO" );
        }
        return contextMenuConfigDAO;
    }

    /**
     * Instantiates the context menu config cache
     * @return cache
     */
    public Cache getContextMenuConfigCache() {
        if( contextMenuConfigCache == null ) {
            throw new GenericSystemException( "Can't get context menu config cache" );
        }
        return contextMenuConfigCache;
    }

}
