/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.eqlext.utils;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.eqlext.ejb.ReportManagerLocal;
import com.queplix.core.modules.eqlext.ejb.ReportManagerLocalHome;
import com.queplix.core.modules.eqlext.jxb.gr.Report;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.cache.CacheObjectManager;

import java.io.OutputStream;
import java.io.PrintWriter;
import java.util.Map;

/**
 * Report builder.
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:42 $
 */

public class ReportBuilder
    implements java.io.Serializable, ReportCallbackHnd {

    // ----------------------------------------------------- variables

    private LogonSession ls;
    private Long processId;
    private Report report;

    private volatile boolean isStarted;
    private volatile int curReqId;
    private volatile int curReqSize;
    private volatile int curPage;

    // ----------------------------------------------------- constructor

    public ReportBuilder( LogonSession ls,
                          Long processId,
                          Report report ) {

        if( ls == null || processId == null || report == null ) {
            throw new NullPointerException( "Incorrect initialization" );
        }

        this.ls = ls;
        this.processId = processId;
        this.report = report;
        this.isStarted = false;
    }

    // ----------------------------------------------------- getters

    public LogonSession getLogonSession() {
        return ls;
    }

    public Long getProcessId() {
        return processId;
    }

    public Report getReport() {
        return report;
    }

    public int getCurReqId() {
        return curReqId;
    }

    public int getCurReqSize() {
        return curReqSize;
    }

    public int getCurPage() {
        return curPage;
    }

    // ----------------------------------------------------- public methods

    /**
     * Start print report procedure.
     * @param transletName translet name
     * @param w output stream writer
     * @throws EQLException
     */
    public void print( String transletName, PrintWriter w )
        throws EQLException {

        print( transletName, null, w );
    }

    /**
     * Start print report procedure.
     * @param transletName translet name
     * @param transletParams translet parameters
     * @param w output stream writer
     * @throws EQLException
     */
    public void print( String transletName, Map transletParams, PrintWriter w )
        throws EQLException {

        // Get ReportManager EJB local interface.
        ReportManagerLocal local = ( ReportManagerLocal ) new CacheObjectManager().
            getLocalObject( JNDINames.ReportManager, ReportManagerLocalHome.class );

        // Set start flag to true.
        isStarted = true;

        // Generate and print report.
        try {
            local.printReport( ls, report, transletName, transletParams, w, this );
        } finally {
            isStarted = false;
        }
    }

    /**
     * Start print report procedure.
     * @param transletName translet name
     * @param transletParams translet parameters
     * @param os output stream
     * @throws EQLException
     */
    public void print( String transletName, Map transletParams, OutputStream os )
        throws EQLException {

        // Get ReportManager EJB local interface.
        ReportManagerLocal local = ( ReportManagerLocal ) new CacheObjectManager().
            getLocalObject( JNDINames.ReportManager, ReportManagerLocalHome.class );

        // Set start flag to true.
        isStarted = true;

        // Generate and print report.
        try {
            local.printReport( ls, report, transletName, transletParams, os, this );
        } finally {
            isStarted = false;
        }
    }

    /**
     * Cancel report procedure
     * @throws EQLException
     */
    public void cancel()
        throws EQLException {

        // Set start flag to false.
        isStarted = false;

        // Remove report.
        ReportHelper.clearDataFile( ls, report );
    }

    /*
     * No javadoc
     * @see ReportCallbackHnd#call
     */
    public void call( ReportCallbackHndEvent ev )
        throws EQLException {

        if( !isStarted ) {
            throw new EQLException( "User canceled print operation..." );
        }

        curReqId = ev.getReqId();
        curPage = ev.getPage();
        if( ev.getIterId() == 0 ) {
            curReqSize = ev.getReqSize();
        }
    }
}
