/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.security;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.integrator.security.AccessRightsManager;
import com.queplix.core.integrator.security.BadNameOrPasswordException;
import com.queplix.core.integrator.security.CryptoHelper;
import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocal;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocalHome;
import com.queplix.core.modules.jeo.gen.UserObject;
import com.queplix.core.modules.jeo.gen.UserObjectHandler;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.StringHelper;
import com.queplix.core.utils.cache.CacheObjectManager;
import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.log.Log;

/**
 * An implementation of LoginModule interface that authenticates 
 * user against QueWeb database
 * @author [AZ] Andrei Zudin
 * @version $Revision: 1.1 $ $Date: 2007/04/12 06:53:01 $
 */

public class DatabaseLoginModule 
    implements LoginModule {
    
    //  Cache object manager
    private static final CacheObjectManager com = new CacheObjectManager();
    
    // Logger.
    private static AbstractLogger logger = Log.getLog(DatabaseLoginModule.class);

    /**
     * Authenticates user by supplied name and password against the database
     *
     * @param loginName user login
     * @param password  not crypted password
     * @return UserObject
     * @throws BadNameOrPasswordException thrown if no such user exists in DB.
     */
    public UserObject doLogin (String loginName, String password) 
        throws BadNameOrPasswordException {
        
        JEOManagerLocal jeoManager = getJEOManager();
        LogonSession ls = AccessRightsManager.getSystemLogonSession();
        UserObject user;

        String passwordDigest = password;
        if(!StringHelper.isEmpty(password)) {
            passwordDigest = CryptoHelper.getDigest(password);
        }
        
        try {
            user = UserObjectHandler.selectByLoginAndPassword(jeoManager, ls, loginName, passwordDigest);
        } catch (EQLException e) {
            logger.ERROR("EQLException: " + e.getMessage(), e);
            throw new GenericSystemException("EQLException: " + e.getMessage());
        }
        if(user == null) {
            throw new BadNameOrPasswordException();
        }
        
        return user;
        
    }
    
    private static JEOManagerLocal getJEOManager() {
        return (JEOManagerLocal) com.getLocalObject(JNDINames.JEOManager, JEOManagerLocalHome.class);
    }
        
    
}
