/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.services;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.modules.services.utils.log.AbstractLogPublisher;
import com.queplix.core.utils.StringHelper;
import com.queplix.core.utils.log.AbstractLogger;

import java.io.Serializable;

/**
 * Base class for custom actions.
 *
 * @see com.queplix.core.modules.services.ActionRunner
 *
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:58 $
 */
public abstract class Action
    extends AbstractLogger
    implements Serializable {

    // ==================================================== Constants and fields

    /** @see StringHelper#EMPTY_VALUE */
    protected static final String EMPTY_STRING = StringHelper.EMPTY_VALUE;

    /** @see StringHelper#EMPTY_NUMBER */
    protected static final int EMPTY_NUMBER = StringHelper.EMPTY_NUMBER;

    // Action context.
    private ActionContext context;

    // ========================================================== Public methods

    /**
     * Initializes the action - call before performing it.
     * @param context action context
     */
    public final void init( ActionContext context ) {
        this.context = context;
    }

    /**
     * The action handler method - override in subclasses.
     * @return result object
     */
    public abstract Serializable perform();

    /**
     * Context getter.
     * @return ActionContext object
     */
    public final ActionContext getContext() {
        if( context == null ) {
            throw new IllegalStateException( "Make initialization first" );
        }
        return context;
    }

    // ============================================================= Log methods

    /**
     * Logs an error message.
     *
     * @param log log publisher to use
     * @param message message to log
     */
    protected final void ERROR( AbstractLogPublisher log, String message ) {
        ERROR( message );
        log.ERROR( message );
        throw new GenericSystemException( message );
    }

    /**
     * Logs an error message.
     *
     * @param log log publisher to use
     * @param message message to log
     * @param t error cause
     */
    protected final void ERROR( AbstractLogPublisher log, String message, Throwable t ) {
        ERROR( message, t );
        log.ERROR( message );
        throw new GenericSystemException( message, t );
    }

    /**
     * Logs a warning.
     *
     * @param log log publisher to use
     * @param message message to log
     */
    protected final void WARN( AbstractLogPublisher log, String message ) {
        WARN( message );
        log.WARN( message );
    }

    /**
     * Logs an informational message.
     *
     * @param log log publisher to use
     * @param message message to log
     */
    protected final void INFO( AbstractLogPublisher log, String message ) {
        INFO( message );
        log.INFO( message );
    }

    /**
     * Logs a debug message.
     *
     * @param log log publisher to use
     * @param message message to log
     */
    protected final void DEBUG( AbstractLogPublisher log, String message ) {
        DEBUG( message );
        log.DEBUG( message );
    }

    // ======================================================= Inherited methods

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return "[" + getClass() + ", context = " + context + "]";
    }

}
