/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.tools;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.integrator.security.AccessRightsManager;
import com.queplix.core.modules.services.Action;
import com.queplix.core.modules.services.ActionContext;
import com.queplix.core.modules.services.ActionRunner;
import com.queplix.core.modules.services.XAActionContext;
import com.queplix.core.utils.cache.CacheObjectManager;
import com.queplix.core.utils.log.AbstractLogger;

import javax.naming.Binding;
import javax.naming.Context;
import javax.naming.LinkRef;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * <p>Base class for all tools</p>
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:14 $
 */

public abstract class AbstractTool
    extends AbstractLogger implements java.io.Serializable {

    // variables
    private final CacheObjectManager com = new CacheObjectManager();

    // temp list for #printBindings()
    private final List printedContextList = new ArrayList();


    /**
     * Get default logon session for any tool
     * @return admin logon session
     */
    protected LogonSession getLogonSession() {
        return AccessRightsManager.getSystemLogonSession();
    }

    /**
     * Get cache manager
     * @return CacheObjectManager object
     */
    protected CacheObjectManager getCacheObjectManager() {
        return com;
    }

    /**
     * Run action
     * @param action Action object
     * @param params action parameters
     * @throws Exception
     */
    protected void callAction( Action action, Map params )
        throws Exception {

        ActionContext context = new ActionContext( getLogonSession() );
        context.addParameters( params );
        ActionRunner.run( action, context );
    }

    /**
     * Run XA action
     * @param action Action object
     * @param params action parameters
     * @throws Exception
     */
    protected void callXAAction( Action action, Map params )
        throws Exception {

        XAActionContext context = new XAActionContext( getLogonSession() );
        context.addParameters( params );
        ActionRunner.runAsXA( action, context );
    }

    //
    // Print all objects binded in context ctx
    //
    protected void printBindings( Context ctx )
        throws NamingException {

        if( ctx == null ) {
            return;
        }

        if( printedContextList.contains( ctx ) ) {
            System.out.println( "Context: " + ctx.getNameInNamespace() + " already printed...skip" );
            return;
        }
        printedContextList.add( ctx );

        System.out.println( "Context: " + ctx.getNameInNamespace() + "\n[" );
        try {
            for( NamingEnumeration ne = ctx.listBindings( "" ); ne != null && ne.hasMore(); ) {
                Binding bnd = ( Binding ) ne.next();
                Object o = bnd.getObject();
                if( o instanceof Context ) {
                    printBindings( ( Context ) o );
                } else {
                    System.out.println( "   Class: " + bnd.getClassName() + ". Binding name: " + bnd.getName() );
                    if( o instanceof LinkRef ) {
                        LinkRef ref = ( LinkRef ) o;
                        System.out.println( "       -> Addr: " + ref.get( 0 ) );
                    }
                }
            }
        } catch( Exception ex ) {
            ex.printStackTrace();
            System.out.println( "Some error occured: " + ex.getMessage() );
        }
        System.out.println( "\n]" );
    }

}
