/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.utils.cache;

import com.queplix.core.utils.dao.InterfaceProperty;

import java.util.Collection;
import java.util.Map;
import java.util.SortedMap;

/**
 * <p>Cache interface</p>
 * @author [ALB] Baranov Andrey
 * @version 1.0
 */

public interface Cache
    extends InterfaceProperty {

    /**
     * Get value from the cache
     * @param key key
     * @return value
     */
    Object get( Object key );

    /**
     * Returns a view of the portion of data whose keys are strictly less than <code>toKey</code>.
     * @param toKey key
     * @return SortedMap
     */
    SortedMap headMap( Comparable toKey );

    /**
     * Returns a view of the portion of data whose keys range from <code>fromKey</code>,
     * inclusive, to <code>toKey</code>, exclusive.
     * @param fromKey key
     * @param toKey key
     * @return SortedMap
     */
    SortedMap subMap( Comparable fromKey, Comparable toKey );

    /**
     * Returns a view of the portion of data whose keys are greater than or equal to <code>fromKey</code>.
     * @param fromKey key
     * @return SortedMap
     */
    SortedMap tailMap( Comparable fromKey );

    /**
     * Returns full view of data.
     * @return SortedMap
     */
    SortedMap toMap();

    /**
     * Checks whether a value with the key specified exists in the cache
     * @param key key
     * @return boolean
     */
    boolean containsKey( Object key );

    /**
     * Get collection of keys.
     * @return collection
     */
    Collection keys();

    /**
     * Returns the last (highest) key.
     * @return Object
     */
    Object getLastKey();

    /**
     * Returns the first (lowest) key.
     * @return Object
     */
    Object getFirstKey();

    /**
     * Get collection of values.
     * @return collection
     */
    Collection values();

    /**
     * Put value to the cache
     * @param key key
     * @param o value
     */
    void put( Object key, Object o );

    /**
     * Put values to the cache
     * @param map Map of key-value pairs
     */
    void put( Map map );

    /**
     * Open cache
     * @return previos state
     */
    boolean open();

    /**
     * Close cache
     */
    void close();

    /**
     * Remove value from the cache
     * @param key key
     */
    void remove( Object key );

    /**
     * Clear cache
     */
    void clear();

    /**
     * Is cache open
     * @return true or false
     */
    boolean isOpen();

    /**
     * Is cache empty
     * @return true or false
     */
    boolean isEmpty();
}
