/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.utils.xml.impl.v26;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.log.Log;
import com.queplix.core.utils.xml.AbstractTransletWrapper;
import org.apache.xalan.Version;

import javax.xml.transform.Result;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.stream.StreamSource;
import java.util.Iterator;
import java.util.Map;

/**
 * <p>Translet wrapper implementation for Xalan</p>
 * @author [BAA] Alexander Balandin
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:27 $
 */

public class XalanTransletWrapper
    extends AbstractTransletWrapper {

    // ----------------------------------------------------- constants

    private static final AbstractLogger logger = Log.getLog( XalanTransletWrapper.class );

    // ----------------------------------------------------- public methods

    /*
     * No javadoc
     * @see TransletWrapper#transform(Source,Result,String,Map)
     */
    public void transform( Source source, Result result, String transletName, Map params ) {

        // Init.
        String packageName = null;
        int pos = transletName.lastIndexOf( "." );
        if( pos > 0 ) {
            packageName = transletName.substring( 0, pos );
            transletName = transletName.substring( pos + 1 );
        }

        if( logger.getLogger().isDebugEnabled() ) {
            logger.DEBUG( "Make XSLT. Translet: " + transletName + ". Class: " +
                          packageName + "\nParams: " + params );

        }

        // Get TransformerFactory.
        TransformerFactory factory = TransformerFactory.newInstance();
        if( logger.getLogger().isDebugEnabled() ) {
            logger.DEBUG( "TransformerFactory class: " + factory.getClass() );
            logger.DEBUG( "Xalan version: " + Version.getVersion() );
        }

        try {
            factory.setAttribute( "use-classpath", Boolean.TRUE );
            if( packageName != null ) {
                factory.setAttribute( "package-name", packageName );
            }

        } catch( IllegalArgumentException ex ) {
            throw new GenericSystemException( "Could not set XSLTC-specific TransformerFactory attributes. " +
                                              "Transformation failed: " + ex.getMessage(), ex );
        }

        try {
            // Get Transformer.
            Transformer t = factory.newTransformer( new StreamSource( transletName ) );

            // Add XSLT global parameters
            t.clearParameters();
            if( params != null ) {
                Iterator it = params.keySet().iterator();
                while( it.hasNext() ) {
                    String key = ( String ) it.next();
                    Object value = params.get( key );
                    t.setParameter( key, value );
                }
            }

            // Do transform.
            t.transform( source, result );

        } catch( TransformerException ex ) {
            throw new GenericSystemException( "Transformer exception: " + ex.getMessage(), ex );

        } catch( Exception ex ) {
            throw new GenericSystemException( "Unknown exception: " + ex.getMessage(), ex );
        }
    }

}
