/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.common.ui;

import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.DialogBox;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.app.vo.uisettings.DialogUISettings;
import com.queplix.core.client.common.ui.resizable.ResizableDialogBox;

/**
 * Anchored dialog box. When shown it automatically positions itself
 * according to the following rules:
 * <ol>
 *     <li>It tries to position its left top corner next to its anchor.
 *         An anchor is a widget passed to the setAnchor() method.
 *     <li>If the dialog box completely fits in the visible area, then
 *         it is shown. Otherwise it is repositioned to fit and then shown.
 * </ol>
 * @author Sultan Tezadov
 * @since 19 Dec 2006
 */
public class AnchoredDialogBox extends DialogBox {
    private static final int POPUP_WINDOW_GAP = 10;
    
    private Widget anchor;
    private Widget widget;
    private int left;
    private int top;
    private boolean isShown;
    protected DialogUISettings uiSettings;

    public AnchoredDialogBox() {
    }
    
    public AnchoredDialogBox(boolean autoHide) {
        super(autoHide);
    }
    
    public AnchoredDialogBox(Widget anchor) {
        this.anchor = anchor;
    }
    
    public AnchoredDialogBox(boolean autoHide, Widget anchor) {
        super(autoHide);
        this.anchor = anchor;
    }
    
    public Widget getAnchor() {
        return anchor;
    }
    
    public void setAnchor(Widget anchor) {
        this.anchor = anchor;
    }
    
    public void show() {
        super.show();
        position();
        // working around IE bug -- browser crashes when positionning
        // dialog with TextArea:
        Widget w = getWidget();
        w.setVisible(false);
        setPopupPosition(left, top);
        w.setVisible(true);
        isShown = true;
    }

    public void hide() {
        super.hide();
        isShown = false;
    }
    
    private void position() {
        int width = getOffsetWidth();
        int height = getOffsetHeight();
        int screenWidth = Window.getClientWidth();
        int screenHeight = Window.getClientHeight();
        if ((uiSettings != null)  &&  uiSettings.isPositionSet()) {
            left = uiSettings.getLeft();
            top = uiSettings.getTop();
        } else if (anchor != null) {
            left = anchor.getAbsoluteLeft();
            top = anchor.getAbsoluteTop() + anchor.getOffsetHeight();
        } else {
            // center
            left = (screenWidth - width) / 2;
            top = (screenHeight - height) / 2;
        }
        if (left + width > screenWidth) {
            left = screenWidth - width;
        }
        if (top + height > screenHeight) {
            top = screenHeight - height;
        }
        if(width > screenWidth){
            if(this instanceof ResizableDialogBox){
                // in this case, pop-up needs to be a bit smaller, than the visible area
                ((ResizableDialogBox)this).resize(screenWidth - width - POPUP_WINDOW_GAP, 0);
            }
        }
        if(height > screenHeight){
            if(this instanceof ResizableDialogBox){
                ((ResizableDialogBox)this).resize(0, screenHeight - height - POPUP_WINDOW_GAP);
            }
        }
        if (left < 0) {
            left = 0;
        }
        if (top < 0) {
            top = 0;
        }
    }
    
    public void onMouseDown(Widget sender, int x, int y) {
        // working around IE bug -- browser crashes when dragging
        // dialog with TextArea:
        widget.setVisible(false);
        super.onMouseDown(sender, x, y);
    }
    
    public void onMouseUp(Widget sender, int x, int y) {
        // working around IE bug -- browser crashes when dragging
        // dialog with TextArea:
        widget.setVisible(true);
        super.onMouseUp(sender, x, y);
        updateUISettings();
    }
    
    public void setWidget(Widget w) {
        this.widget = w;
        super.setWidget(w);
    }
    
    public Widget getWidget() {
        return widget;
    }
    
    public boolean isDialogShown() {
        return isShown;
    }
    
    public DialogUISettings getUISettings() {
        return uiSettings;
    }
    
    public void setUISettings(DialogUISettings uiSettings) {
        this.uiSettings = uiSettings;
    }
    
    private void updateUISettings() {
        int newLeft = getAbsoluteLeft();
        int newTop = getAbsoluteTop();
        if ((left != newLeft)  ||  (top != newTop)) {
            if (uiSettings == null) {
                uiSettings = new DialogUISettings();
            }
            uiSettings.setLeft(newLeft);
            uiSettings.setTop(newTop);
        }
    }
}
