/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT 
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the 
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.common.ui.resizable;

import com.google.gwt.user.client.ui.*;
import com.queplix.core.client.common.StringUtil;
import com.queplix.core.client.common.ui.QTabPanelBase;
import com.queplix.core.client.common.ui.QTabBar;

/**
 * Resizable tab panel.
 * @author Sultan Tezadov
 * @since 31 Dec 2006
 */
public class ResizableTabPanel extends QTabPanelBase implements Resizable {
    
    public ResizableTabPanel() {
        this.addTabListener(new TabListener() {
            public boolean onBeforeTabSelected(SourcesTabEvents sender, int tabIndex) {
                return true;
            }
            public void onTabSelected(SourcesTabEvents sender, int tabIndex) {
                tabSelected(tabIndex);
            }
        });
    }

    private void checkResizable(Widget w) {
        if (! (w instanceof Resizable)) {
            throw new IllegalArgumentException(
                    "Illegal argument: w. It must implement Resizable.");
        }
    }

    private void tabSelected(int tabIndex) {
        doSetChildOffsetHeight(tabIndex);
        doSetChildOffsetWidth(tabIndex);
    }
    
    public void insert(Widget widget, String tabText, boolean asHTML, int beforeIndex) {
        checkResizable(widget);
        super.insert(widget, tabText, asHTML, beforeIndex);
    }

    // ======================== Resizable implementation =====================
    private int offsetHeight;
    private int offsetWidth;
    
    public void setOffsetHeight(int height) {
        if ((height < 0)  ||  (offsetHeight == height)) return; // invalid or unchanged value
        offsetHeight = height;
        if (isAttached()) {
            doSetOffsetHeight();
        }
    }

    public void setOffsetWidth(int width) {
        if ((width < 0)  ||  (offsetWidth == width)) return; // invalid or unchanged value
        offsetWidth = width;
        if (isAttached()) {
            doSetOffsetWidth();
        }
    }
    
    private void doSetOffsetHeight() {
        int index = getTabBar().getSelectedTab();
        doSetChildOffsetHeight(index);
        int deckHeight = getDeckOffsetHeight();
        getDeckPanel().setHeight(StringUtil.pixelToSize(deckHeight));
        super.setHeight(StringUtil.pixelToSize(offsetHeight));
    }

    private void doSetChildOffsetHeight(int tabIndex) {
        Resizable widget = (Resizable) getDeckPanel().getWidget(tabIndex);
        if (widget != null) {
            widget.setOffsetHeight(getDeckOffsetHeight());
        }
    }
    
    private int getDeckOffsetHeight() {
        int tabBarHeight = getTabBar().getOffsetHeight();
        int deckHeight = offsetHeight - tabBarHeight;
        deckHeight = (deckHeight > 0) ? deckHeight : 0;
        return deckHeight;
    }

    private void doSetOffsetWidth() {
        int index = getTabBar().getSelectedTab();
        doSetChildOffsetWidth(index);
        String offsetWidthSize = StringUtil.pixelToSize(offsetWidth);
        getDeckPanel().setWidth(offsetWidthSize);
        super.setWidth(offsetWidthSize);
    }
    
    private void doSetChildOffsetWidth(int tabIndex) {
        Resizable widget = (Resizable) getDeckPanel().getWidget(tabIndex);
        if (widget != null) {
            widget.setOffsetWidth(offsetWidth);
        }
    }
    
    protected void onAttach() {
        super.onAttach();
        if (offsetHeight > 0) { // offsetHeight was set
            doSetOffsetHeight();
        }
        if (offsetWidth > 0) { // offsetWidth was set
            doSetOffsetWidth();
        }
    }

    public void setHeight(String height) {
        setOffsetHeight(StringUtil.sizeToPixel(height));
    }

    public void setWidth(String width) {
        setOffsetWidth(StringUtil.sizeToPixel(width));
    }
    // ===================== End of Resizable implementation ==================

}
