/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.common.ui.upload;

import com.google.gwt.user.client.ui.Button;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FileUpload;
import com.google.gwt.user.client.ui.FormHandler;
import com.google.gwt.user.client.ui.FormPanel;
import com.google.gwt.user.client.ui.FormSubmitCompleteEvent;
import com.google.gwt.user.client.ui.FormSubmitEvent;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.ui.DialogHelper;

/**
 * QFileUpload.
 * @author Sultan Tezadov
 * @since 28 Feb 2007
 */
public class QFileUpload extends Composite implements 
        FormHandler,
        ClickListener,
        SourcesQFileUploadEvents
{
    private FormPanel form;
    private FileUpload fileUpload;
    private Button uploadButton;
    private QFileUploadListenerCollection listeners;
    
    public static final String DEFAULT_ACTION = "../fileUpload/new";
    
    private static final String FILE_LABEL = "File";
    private static final String ERR_MSG_FILE_IS_EMPTY = "Field '" + FILE_LABEL + "' is empty.";

    public QFileUpload() {
        this(DEFAULT_ACTION, true);
    }
    
    public QFileUpload(String action) {
        this(action, true);
    }
    
    public QFileUpload(String action, boolean initalizeUploadButton) {
        form = new FormPanel();
        form.setWidth("100%");
        if (action == null) {
            setAction(DEFAULT_ACTION);
        } else {
            setAction(action);
        }
        form.setEncoding(FormPanel.ENCODING_MULTIPART);
        form.setMethod(FormPanel.METHOD_POST);
        
        Label fileLabel = new Label(FILE_LABEL + ": ");
        
        fileUpload = new FileUpload();
        fileUpload.setName("file");
        fileUpload.setWidth("600px");
        
        form.addFormHandler(this);
        
        HorizontalPanel uploadPanel = new HorizontalPanel();
        uploadPanel.add(fileLabel);
        uploadPanel.add(fileUpload);
        if (initalizeUploadButton) {
            uploadButton = new Button("Upload", this);
            uploadPanel.add(uploadButton);
        }
        
        form.setWidget(uploadPanel);
        initWidget(form);
    }
    
    public void setAction(String action) {
        form.setAction(action);
    }

    public void onSubmitComplete(FormSubmitCompleteEvent event) {
        String res = event.getResults();
        if (res.indexOf("|RESULT=SUCCESS|") != -1) {
            res = res.split("[|]{1}")[1];
            if (listeners != null) {
                listeners.fireFileUploaded(this, res);
            }
        } else {
            DialogHelper.showModalErrorDialog(res, null);
        }
    }
    
    public void onSubmit(FormSubmitEvent event) {
        boolean fn = fileUpload.getFilename().trim().equals("");
        if (fn) {
            DialogHelper.showModalMessageDialog(ERR_MSG_FILE_IS_EMPTY);
            event.setCancelled(true);
        }
    }
    
    public void onClick(Widget sender) {
        submitUpload();
    }
    
    public void submitUpload() {
        form.submit();
    }
    
    public void addQFileUploadListener(QFileUploadListener listener) {
        if (listeners == null) {
            listeners = new QFileUploadListenerCollection();
        }
        listeners.add(listener);
    }

    public void removeQFileUploadListener(QFileUploadListener listener) {
        if (listeners != null) {
            listeners.remove(listener);
        }
    }

    public void setUploadButtonVisible(boolean visible) {
        uploadButton.setVisible(visible);
    }
    
    public void setUploadButtonCaption(String caption) {
        uploadButton.setText(caption);
    }

    public String getFileUploadName() {
        return fileUpload.getFilename();
    }

    public String getAction() {
        return form.getAction();
    }
}
