/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.controls.informgrid;

import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.app.vo.FieldMeta;
import com.queplix.core.client.app.vo.FieldOnDemandData;
import com.queplix.core.client.app.vo.FormButtonMeta;
import com.queplix.core.client.app.vo.FormMeta;
import com.queplix.core.client.app.vo.GridData;
import com.queplix.core.client.app.vo.GridMeta;
import com.queplix.core.client.app.vo.RecordFilter;
import com.queplix.core.client.app.vo.RowData;
import com.queplix.core.client.common.CollectionsHelper;
import com.queplix.core.client.common.event.Event;
import com.queplix.core.client.common.event.EventSource;
import com.queplix.core.client.common.ui.OkayCancelPopup;
import com.queplix.core.client.controls.form.QForm;
import com.queplix.core.client.controls.form.QFormController;
import com.queplix.core.client.controls.form.QFormLayout;
import com.queplix.core.client.controls.grid.QGrid;
import com.queplix.core.client.controls.grid.QGridController;
import com.queplix.core.client.controls.grid.QGridModel;

import java.util.List;

/**
 * This class responsible for creating sets of filters for the inner entity in
 * in-form grid control. Finally should return
 * {@link com.queplix.core.client.app.vo.RecordFilter} object.
 *
 * @author Sergey Kozmin
 * @since 03.04.2007
 */
class FilteringDialog extends OkayCancelPopup {
    private QGrid grid;
    private QForm form;

    public FilteringDialog(String caption, GridMeta filteringGrid,
                           FormMeta filteringForm) {
        super(caption);
        initGUI(filteringGrid, filteringForm);
    }

    private void initGUI(GridMeta filteringGrid, FormMeta formMeta) {
        VerticalPanel mainPanel = new VerticalPanel();

        mainPanel.add(new Label("Result"));
        grid = new QGrid(filteringGrid, true, true, false, true, true);
        mainPanel.add(grid.getView());

        mainPanel.add(new Label("Filters"));

        if(formMeta.getLayoutMeta() == null) {
            FieldMeta[] fieldMetas = formMeta.getEntityMeta().getFields();
            int formWidth = 2;

            formMeta.setButtons(getSearchAndClearIfAny(formMeta.getButtons()));

            form = new QForm(formMeta, formWidth, (int) Math.ceil(
                    fieldMetas.length / (float) formWidth),
                    QFormLayout.HORIZONTAL_FLOW_LAYOUT);
        } else {
            form = new QForm(formMeta);
        }

        mainPanel.add(form.getView());
        mainPanel.setCellWidth(form.getView(), "100%");
        form.getFormController().getEventSource().addEventListener(this);
        setWidget(mainPanel);
    }

    private FormButtonMeta[] getSearchAndClearIfAny(FormButtonMeta[] buttons) {
        FormButtonMeta searchBtn = null;
        FormButtonMeta clearBtn = null;
        for(int i = 0; i < buttons.length; i++) {
            if(QFormController.FORM_SEARCH_BUTTON.equalsIgnoreCase(buttons[i].getId())) {
                searchBtn = buttons[i];
            } else if(QFormController.FORM_CLEAR_BUTTON.equalsIgnoreCase(buttons[i].getId())) {
                clearBtn = buttons[i];
            }
        }
        int len = ((searchBtn == null) ? 0 : 1) + ((clearBtn == null) ? 0 : 1);
        FormButtonMeta[] formButtonMetas = new FormButtonMeta[len];

        if(clearBtn != null) {
            formButtonMetas[--len] = clearBtn;
        }
        if(searchBtn != null) {
            formButtonMetas[--len] = searchBtn;
        }
        return formButtonMetas;
    }

    public void setGridData(List data) {
        RowData[] array = new RowData[data.size()];
        CollectionsHelper.copyToArray(data, array);
        QGridModel qGridModel = grid.getModel();
        qGridModel.setGridData(new GridData(array, ""));
    }

    public RecordFilter getDialogFilter() {
        List markedRows = grid.getController().getMarkedRecordsIds();
        RecordFilter filter;
        if(markedRows.size() > 0) {
            filter = RecordFilter.createRecordsFilter(markedRows);
        } else {
            List nonEmptyFilters = form.getModel().getNonEmptyFilters();
            filter = RecordFilter.createFieldsFilter(nonEmptyFilters);
        }
        return filter;
    }

    public void setFieldOnDemandData(FieldOnDemandData data) {
        form.getModel().setOnDemandDataForElement(data);
    }

    public void onEvent(Event event, Widget sender) {
        if(sender == this) {//handle popup events
            if(OkayCancelPopup.Events.OK == event
                    || OkayCancelPopup.Events.CANCEL == event) {
                super.onEvent(event, sender);
                filteringEventSource.fireEvent(event);//redirect OK and CANCEL events
            }
        } else if(sender == form.getView()) {//handle form events
            if(QFormController.Events.FORM_SEARCH_BUTTON_EVENT == event ||
                    QFormController.Events.FORM_CONTROL_NEED_MORE_DATA_EVENT == event) {
                filteringEventSource.fireEvent(event);
            } else if(QFormController.Events.FORM_CLEAR_BUTTON_EVENT == event) {
                grid.getController().clearGrid();
                form.getFormController().clearForm();
            }
        } else if(sender == grid.getView()) {
            if(QGridController.Events.RECORD_SELECTED == event) {
                filteringEventSource.fireEvent(event);
            }
        }
    }

    // -------------------- public events ------------------------
    public EventSource getFilteringEventSource() {
        return filteringEventSource;
    }

    private EventSource filteringEventSource = new EventSource(this);

    public static interface Events extends OkayCancelPopup.Events {
        Event/*]<FieldDataRequest>[*/ FORM_CONTROL_NEED_MORE_DATA_EVENT
                = QFormController.Events.FORM_CONTROL_NEED_MORE_DATA_EVENT;

        Event FORM_SEARCH_BUTTON_EVENT = QFormController.Events.FORM_SEARCH_BUTTON_EVENT;
    }
    // ----------------- end of public events --------------------
}
