/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.controls.informgrid;

import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.DockPanel;
import com.google.gwt.user.client.ui.Grid;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.app.vo.FieldDataRequest;
import com.queplix.core.client.app.vo.FormMeta;
import com.queplix.core.client.app.vo.GetControlInformOnDemandData;
import com.queplix.core.client.app.vo.GetRecordsInformOnDemandData;
import com.queplix.core.client.app.vo.GridMeta;
import com.queplix.core.client.app.vo.InFormGridFieldMeta;
import com.queplix.core.client.app.vo.InFormGridOnDemandData;
import com.queplix.core.client.app.vo.InFormGridType;
import com.queplix.core.client.app.vo.RecordFilter;
import com.queplix.core.client.app.vo.SearchInformOnDemandData;
import com.queplix.core.client.common.event.Event;
import com.queplix.core.client.common.event.EventListener;
import com.queplix.core.client.common.ui.grid.DataGrid;
import com.queplix.core.client.common.ui.grid.DefaultGridView;
import com.queplix.core.client.controls.QButton;
import com.queplix.core.client.controls.QFormElement;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

class QInformGridViewImpl extends QInformGridView implements ClickListener,
        EventListener {
    public final static String GRID_CONTAINER_STYLE_NAME = QInformGrid.INFORMGRID_STYLE_NAME_PREFIX + "container";
    public final static String BTN_PANEL_STYLE_NAME = QInformGrid.INFORMGRID_STYLE_NAME_PREFIX + "btnPanel";
    public final static String SCROLL_PANEL_STYLE_NAME = QInformGrid.INFORMGRID_STYLE_NAME_PREFIX + "scrollPanel";
    public final static String GRID_ELEMENT_STYLE_NAME = QInformGrid.INFORMGRID_STYLE_NAME_PREFIX + "gridElement";
    public final static String GRID_HEADER_ELEMENT_STYLE_NAME = QInformGrid.INFORMGRID_STYLE_NAME_PREFIX + "gridHeaderElement";

    public static final String LINK_BUTTON_CAPTION = "Link";
    public static final String UNLINK_BUTTON_CAPTION = "Unlink";
    public static final String FILTERS_BUTTON_CAPTION = "Set Filters";//todo add captions

    private QInformGridModelImpl model;

    private ArrayList actionsListeners = new ArrayList();

    //view related
    private Grid mainPanel;

    private VerticalPanel gridPanel;
    private DataGrid informGrid;

    private HorizontalPanel btnPanel;
    private QButton linkBtn;
    private QButton unlinkBtn;
    private QButton filtersBtn;
    private FilteringDialog filteringDialog;

    public QInformGridViewImpl(QInformGridModelImpl model) {
        super(model, QFormElement.HORIZONTAL);
        this.model = model;
        setupWidgets();
        initListeners();
    }

    private void initListeners() {
        linkBtn.addClickListener(this);
        unlinkBtn.addClickListener(this);
        filtersBtn.addClickListener(this);
        informGrid.getEventSource().addEventListener(this);
    }

    private void setupWidgets() {
        mainPanel = new Grid(3, 3);
        mainPanel.setCellPadding(0);
        mainPanel.setCellSpacing(0);

        DockPanel titleBar = new DockPanel();
        titleBar.add(captionLabel, DockPanel.WEST);
        captionLabel.setStyleName("form_topSide");
        captionLabel.setWordWrap(false);
        captionLabel.setWidth("100%");
        titleBar.setCellWidth(captionLabel, "100%");
//        titleBar.setWidth("100%");don't do that

        Label leftTop = new Label(" ");
        leftTop.setStyleName("form_leftTop");

        Label rightTop = new Label(" ");
        rightTop.setStyleName("form_rightTop");

        Label leftSide = new Label(" ");
        leftSide.setStyleName("form_leftSide");

        Label leftBottom = new Label(" ");
        leftBottom.setStyleName("form_leftBottom");

        Label topSide = new Label(" ");
        topSide.setStyleName("form_topSide");

        Label bottomSide = new Label(" ");
        bottomSide.setStyleName("form_bottomSide");

        Label rightSide = new Label(" ");
        rightSide.setStyleName("form_rightSide");

        Label rightBottom = new Label(" ");
        rightBottom.setStyleName("form_rightBottom");

        gridPanel = new VerticalPanel();
        gridPanel.setSpacing(0);
        gridPanel.addStyleName(GRID_CONTAINER_STYLE_NAME);

        informGrid = new DataGrid(model.getGridModel(),
                DefaultGridView.INSTANCE, false, false);

        linkBtn = new QButton(LINK_BUTTON_CAPTION);
        unlinkBtn = new QButton(UNLINK_BUTTON_CAPTION);
        filtersBtn = new QButton(FILTERS_BUTTON_CAPTION);

        btnPanel = new HorizontalPanel();
        btnPanel.setHorizontalAlignment(HorizontalPanel.ALIGN_RIGHT);
        btnPanel.setStyleName(BTN_PANEL_STYLE_NAME);

        Label empLablel = new Label();
        btnPanel.add(empLablel);
        btnPanel.add(linkBtn);
        btnPanel.add(unlinkBtn);
        btnPanel.add(filtersBtn);

        btnPanel.setCellWidth(empLablel, "100%");

        gridPanel.add(informGrid);
        gridPanel.add(btnPanel);

        mainPanel.setWidget(0, 0, leftTop);
        mainPanel.setWidget(0, 1, titleBar);
        mainPanel.setWidget(0, 2, rightTop);
        mainPanel.setWidget(1, 0, leftSide);
        mainPanel.setWidget(1, 1, gridPanel);
        mainPanel.setWidget(1, 2, rightSide);
        mainPanel.setWidget(2, 0, leftBottom);
        mainPanel.setWidget(2, 1, bottomSide);
        mainPanel.setWidget(2, 2, rightBottom);

        initWidget(mainPanel);
    }

    public void setButtonsEnabled(int buttonIndex, boolean isEnabled) {
        getButton(buttonIndex).setEnabled(isEnabled);
    }

    /**
     * Can be performed for single button.
     * @param buttonIndex index of single button.
     * @param isVisible visible state
     */
    public void setButtonVisible(int buttonIndex, boolean isVisible) {
        getButton(buttonIndex).setVisible(isVisible);
    }

    private QButton getButton(int buttonIndex) {
        QButton ret = null;
        if ((buttonIndex & QInformGrid.LINK_BUTTON_INDEX) != 0) {
            ret = linkBtn;
        }
        if ((buttonIndex & QInformGrid.UNLINK_BUTTON_INDEX) != 0) {
            ret = unlinkBtn;
        }
        if ((buttonIndex & QInformGrid.FILTERS_BUTTON_INDEX) != 0) {
            ret = filtersBtn;
        }
        return ret;
    }

    /*public boolean askUserQuestion(String question) {
        return DialogHelper.showModalQuestionDialog(question) == DialogHelper.YES;
    }*/

    public void onClick(Widget sender) {
        if(sender.equals(linkBtn)) {
            fireAction(QInformGrid.LINK_BUTTON_INDEX);
        } else if(sender.equals(unlinkBtn)) {
            fireAction(QInformGrid.UNLINK_BUTTON_INDEX);
        } else if(sender.equals(filtersBtn)) {
            fireAction(QInformGrid.FILTERS_BUTTON_INDEX);
        }
    }

    protected void onModeChanged(int newMode) {
        fireAction(QInformGrid.CONTROL_STATE_CHANGED_INDEX);
    }

    protected void setEnabled(boolean isEnabled) {
        //do nothing, do all in method onModeChanged
    }

    protected void setClientWidth(String clientWidth) {
        mainPanel.setWidth(clientWidth);
        informGrid.setWidth(clientWidth);
        gridPanel.setWidth(clientWidth);
        btnPanel.setWidth(clientWidth);
    }

    public void onDemandModelChanged() {
        super.onDemandModelChanged();

        InFormGridOnDemandData gridOnDemandData = model.getOnDemandData();
        int type = gridOnDemandData.getInformGridRequestType();
        switch(type) {
            case InFormGridType.SEARCH_TYPE: {
                SearchInformOnDemandData data = (SearchInformOnDemandData) gridOnDemandData;
                filteringDialog.setGridData(data.getData());
                break;
            }
            case InFormGridType.GET_CONTROL_DATA_TYPE: {
                GetControlInformOnDemandData data = (GetControlInformOnDemandData) gridOnDemandData;
                filteringDialog.setFieldOnDemandData(data.getData());
                break;
            }
            case InFormGridType.GET_RECORDS_DATA_TYPE: {
                GetRecordsInformOnDemandData data = (GetRecordsInformOnDemandData) gridOnDemandData;
                fireAddRowsToModel(data.getData(), data.getFilters());
                break;
            }
        }
    }                                       

    public int getClientWidth() {
        return mainPanel.getOffsetWidth();
    }

    public void addActionListener(ViewActionsListener listener) {
        actionsListeners.add(listener);
    }

    public void removeActionListener(ViewActionsListener listener) {
        actionsListeners.remove(listener);
    }

    private void fireAction(int action) {
        for (int i = 0; i < actionsListeners.size(); i++) {
            ViewActionsListener listener = (ViewActionsListener) 
                    actionsListeners.get(i);
            listener.onAction(action);
        }
    }

    private void fireOnSearch(RecordFilter filter) {
        for (int i = 0; i < actionsListeners.size(); i++) {
            ViewActionsListener listener = (ViewActionsListener)
                    actionsListeners.get(i);
            listener.onSearchRequest(filter);
        }
    }

    private void fireOnMoreData(FieldDataRequest filter) {
        for (int i = 0; i < actionsListeners.size(); i++) {
            ViewActionsListener listener = (ViewActionsListener)
                    actionsListeners.get(i);
            listener.onMoreDataRequest(filter);
        }
    }

    private void fireAddRowsToModel(List rows, Collection filters) {
        for (int i = 0; i < actionsListeners.size(); i++) {
            ViewActionsListener listener = (ViewActionsListener)
                    actionsListeners.get(i);
            listener.addRecordsToModel(rows, filters);
        }
    }

    public void showFilteringDialog() {
        FilteringDialog dialog = getFilteringDialog();
        dialog.show();
    }

    private FilteringDialog getFilteringDialog() {
        if(filteringDialog == null) {
            InFormGridFieldMeta fieldMeta = model.getMeta();
            String caption = "Select filters for [" + fieldMeta.getCaption()
                    + "]";
            GridMeta gridMeta = fieldMeta.getFilteringGridMeta();
            FormMeta formMeta = fieldMeta.getFilteringFormMeta();
            filteringDialog = new FilteringDialog(caption, gridMeta, formMeta);
            filteringDialog.getFilteringEventSource().addEventListener(this);
        }
        return filteringDialog;
    }

    public void onEvent(Event event, Widget sender) {
        if(FilteringDialog.Events.OK == event) {
            fireAction(QInformGrid.FILTERS_WERE_SET_INDEX);
        } else if(FilteringDialog.Events.FORM_CONTROL_NEED_MORE_DATA_EVENT == event) {
            fireOnMoreData((FieldDataRequest)event.getData());
        } else if(FilteringDialog.Events.FORM_SEARCH_BUTTON_EVENT == event) {
            fireOnSearch(filteringDialog.getDialogFilter());
        } else if(DataGrid.Events.GRID_ROW_SELECTED == event) {
            fireAction(QInformGrid.RECORD_SELECTION_CHANGED_INDEX);
        } else if(DataGrid.Events.DELETE_KEY_PRESSED == event) {
            fireAction(QInformGrid.UNLINK_BUTTON_INDEX);
        }
    }

    /**
     * @return List<Integer> of selected indexes
     */
    public List getSelectedRecords() {
        return informGrid.getSelectionController().getSelectedRows();
    }

    public RecordFilter getDialogFilter() {
        return filteringDialog.getDialogFilter();
    }
}
