/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.qwoss.notification;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;

import javax.ejb.CreateException;

import com.queplix.core.jxb.entity.Entity;
import com.queplix.core.modules.config.ejb.EntityViewConfigManagerLocal;
import com.queplix.core.modules.config.ejb.EntityViewConfigManagerLocalHome;
import com.queplix.core.modules.config.ejb.FocusConfigManagerLocal;
import com.queplix.core.modules.config.ejb.FocusConfigManagerLocalHome;
import com.queplix.core.modules.eql.CompoundKey;
import com.queplix.core.modules.eql.ejb.EQLManagerLocal;
import com.queplix.core.modules.eql.ejb.EQLManagerLocalHome;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocal;
import com.queplix.core.modules.jeo.ejb.JEOManagerLocalHome;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.integrator.security.SecurityHelper;
import com.queplix.core.integrator.security.User;
import com.queplix.core.integrator.security.WorkGroup;

import com.queplix.core.utils.DateHelper;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.ejb.AbstractSessionEJB;
import com.queplix.qwoss.utils.ApplicationHelper;
import com.queplix.qwoss.utils.CustomJNDINames;

/**
 * Base class for Escalation Manager EJBs.
 * Subclasses provides EJBs for different entities (e.g. 'case' or 'solution').
 *
 * @author [ALB] Baranov Andrey
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @version $Revision: 1.3.4.1 $ $Date: 2006/07/10 16:09:27 $
 */
abstract class AbstractEntityEscalationManagerEJB
    extends AbstractSessionEJB {

    // ========================================================== Public methods

    /**
     * Generates new escalation events.
     *
     * @param ls user logon session
     * @param entityJEO entity JEO interface instance
     */
    public abstract void generateEvents( LogonSession ls, Object entityJEO );

    /**
     * Fires escalation events.
     * @return number of fired events
     */
    public abstract int fireEvents();

    // ======================================================= Protected methods

    /**
     * Calculates the escalation event's next check date.
     *
     * @param delay delay
     * @param calendarField the delay measurement unit
     * @return next check date
     */
    protected Date getNextCheckDate( Long delay, int calendarField ) {
        if( delay == null ) {
            throw new IllegalStateException( "Delay is null." );
        }
        Date now = DateHelper.getNowDate();
        Calendar calendar = new GregorianCalendar();
        calendar.setTime( now );
        calendar.add( calendarField, delay.intValue() );
        return calendar.getTime();
    }

    protected Date getNextCheckDateForWorkDays( Long delay ) {
        if( delay == null ) {
            throw new IllegalStateException( "Delay is null." );
        }
        int lim = delay.intValue();
        Date now = DateHelper.getNowDate();
        Calendar calendar = new GregorianCalendar();
        calendar.setTime( now );
        for(int i=0;i<lim;i++){
            calendar.add( Calendar.DATE, 1 );
            switch(calendar.get(Calendar.DAY_OF_WEEK)){
                case Calendar.SATURDAY:
                case Calendar.SUNDAY:
                    lim++;
                    break;
            }
        }
        return calendar.getTime();
    }    
    
    /**
     * Loads the group by ID.
     *
     * @param groupID group ID
     * @return group value object
     */
    protected WorkGroup loadGroup(LogonSession ls, Long groupID ) {
        try {
            if( groupID != null ) 
                return SecurityHelper.loadGroup(ls, groupID);
        } catch( EQLException ex ) {
            throwException( ex );
        }
        return null;
    }

    /**
     * Loads the employee user by ID.
     *
     * @param employeeID user ID
     * @return user value object
     */
    protected User loadEmployee(LogonSession ls, Long employeeID ) {
        try {
            if( employeeID != null ) 
                return ApplicationHelper.loadEmployee(ls, employeeID);
        } catch( EQLException ex ) {
            throwException( ex );
        }
        return null;
    }
    
    /**
     * JEO Manager EJB local interface getter.
     * @return EJB local interface reference
     */
    protected JEOManagerLocal getJEOManagerLocal() {
        return( JEOManagerLocal ) getLocalObject( JNDINames.JEOManager, JEOManagerLocalHome.class );
    }

    /**
     * EQL Manager EJB local interface getter.
     * @return EJB local interface reference
     */
    protected EQLManagerLocal getEQLManagerLocal() {
        return( EQLManagerLocal ) getLocalObject( JNDINames.EQLManager, EQLManagerLocalHome.class );
    }
    
    /**
     * EntityViewConfig Manager EJB local interface getter.
     * @return EJB local interface reference
     */
    protected EntityViewConfigManagerLocal getEntityViewConfigManagerLocal() {
        return( EntityViewConfigManagerLocal ) getLocalObject(
            JNDINames.EntityViewConfigManager,
            EntityViewConfigManagerLocalHome.class );
    }

    /**
     * FocusConfig Manager EJB local interface getter.
     * @return EJB local interface reference
     */
    protected FocusConfigManagerLocal getFocusConfigManagerLocal() {
        return( FocusConfigManagerLocal ) getLocalObject( JNDINames.FocusConfigManager, FocusConfigManagerLocalHome.class );
    }

    /**
     * Notification Manager EJB local interface getter.
     * @param ls user session
     * @param entity Entity object
     * @param key CompoundKey object
     * @return EJB local interface reference
     */
    protected NotificationManagerLocal getNotificationManagerLocal( LogonSession ls, Entity entity, CompoundKey key ) {

        NotificationManagerLocalHome home = ( NotificationManagerLocalHome )
            getLocalHome( CustomJNDINames.NotificationManager, NotificationManagerLocalHome.class );

        try {
            return home.create( ls, entity, key );
        } catch( CreateException ex ) {
            throwException( "Can't get NotificationManager local interface", ex );
            return null;
        }
    }

}
