/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.app.vo;

import java.util.ArrayList;
import java.util.List;

/**
 * Entity reference request.
 *
 * @author Sergey Kozmin
 * @since 08.11.2006, 19:27:59
 */
public class EntityReferenceDataRequest extends BaseFieldDataRequest {
    private String entityName = "";
    private String filter;
    private Long id;

    private long page;
    private int pageSize;
    private boolean counterToggledOn;
    private SortField sortOrder;
    /**
     * List of {@link com.queplix.core.client.app.vo.EntityData} objects
     */
    private List additionalFilters;

    /**
     * additional eql filter that will be included to eql request
     */
    private String eqlFilter;

    public EntityReferenceDataRequest(String elementID, String filter, Long id,
                                      String entityName, long page,
                                      int pageSize,
                                      boolean counterToggledOn,
                                      SortField sortOrder,
                                      List additionalFilters,
                                      String eqlFilter) {
        super(FieldType.ENTITYREFERENCE, elementID);
        this.eqlFilter = eqlFilter;
        this.entityName = entityName;
        this.id = id;
        this.filter = filter;
        this.page = page;
        this.pageSize = pageSize;
        this.counterToggledOn = counterToggledOn;
        this.sortOrder = sortOrder;
        if(additionalFilters == null) {
            this.additionalFilters = new ArrayList();
        } else {
            this.additionalFilters = additionalFilters;
        }
    }

    public EntityReferenceDataRequest(String elementID, String filter, Long id,
                                      String entityName, long page,
                                      int pageSize,
                                      boolean counterToggledOn,
                                      SortField sortOrder,
                                      List additionalFilters) {
        this(elementID, filter, id, entityName, page, pageSize,
                counterToggledOn, sortOrder, additionalFilters, null);
    }

    public EntityReferenceDataRequest() {
        this("", "default element id", null, "default entity name", 0, 20,
                false, null, null);
    }

    public void init(String filter, Long id, long page, int pageSize,
                     boolean counterToggledOn, SortField sortOrder) {
        this.filter = filter;
        this.id = id;
        this.page = page;
        this.pageSize = pageSize;
        this.counterToggledOn = counterToggledOn;
        this.sortOrder = sortOrder;
    }

    public String getFilter() {
        return filter;
    }

    public void setFilter(String filter) {
        this.filter = filter;
    }

    public Long getId() {
        return id;
    }

    public long getPage() {
        return page;
    }

    public void setPage(long page) {
        this.page = page;
    }

    public int getPageSize() {
        return pageSize;
    }

    public void setPageSize(int pageSize) {
        this.pageSize = pageSize;
    }

    public boolean isCounterToggledOn() {
        return counterToggledOn;
    }

    public void setCounterToggledOn(boolean counterToggledOn) {
        this.counterToggledOn = counterToggledOn;
    }

    public SortField getSortOrder() {
        return sortOrder;
    }

    public void setSortOrder(SortField sortOrder) {
        this.sortOrder = sortOrder;
    }

    public String getEntityName() {
        return entityName;
    }

    public void setEntityName(String entityName) {
        this.entityName = entityName;
    }

    /**
     * Could not return null
     *
     * @return additional filters for request
     */
    public List getAdditionalFilters() {
        return additionalFilters;
    }

    /**
     * Set single additional filter for the requesting entity.
     *
     * @param data single field filter
     */
    public void setAdditionalFilterForRequestingEntity(FieldData data) {
        setAdditionalFilter(data, getEntityName());
    }

    /**
     * Set single filter for the filteringEntity
     *
     * @param data            single filter for filtering entity
     * @param filteringEntity filtering entity id.
     */
    public void setAdditionalFilter(FieldData data, String filteringEntity) {
        setAdditionalFilter(new EntityData(filteringEntity, new Long(-1L),
                new FieldData[]{data}));
    }

    /**
     * Set single filter for an entity.
     *
     * @param data filtering entity data. No need entity id in this object.
     */
    public void setAdditionalFilter(EntityData data) {
        additionalFilters.clear();
        additionalFilters.add(data);
    }

    /**
     * Adds filter to the existing filters.
     *
     * @param data            field entity
     * @param filteringEntity filtering entity id.
     */
    public void addAdditionalFilter(FieldData data, String filteringEntity) {
        addAdditionalFilter(new EntityData(filteringEntity, new Long(-1L),
                new FieldData[]{data}));
    }

    /**
     * Adds filter to the existing filters. Filter entity will be taken from the requesting entity.
     *
     * @param data field filter.
     */
    public void addAdditionalFilterForRequestingEntity(FieldData data) {
        addAdditionalFilter(data, getEntityName());
    }

    /**
     * Adds filter for entity.
     *
     * @param data entity filter.
     */
    public void addAdditionalFilter(EntityData data) {
        additionalFilters.add(data);
    }

    /**
     * Clears additional filters.
     */
    public void clearAdditionalFilter() {
        additionalFilters.clear();
    }

    public String getEqlFilter() {
        return eqlFilter;
    }

    public void setEqlFilter(String eqlFilter) {
        this.eqlFilter = eqlFilter;
    }
}
