/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.app.vo;

import com.google.gwt.user.client.rpc.IsSerializable;
import com.queplix.core.client.common.StringUtil;

import java.util.Arrays;

/**
 * Represent single row element, that has it's own internal id and array of strings values to be shown in grid.
 *
 * @author Sergey Kozmin
 * @since 18.10.2006, 12:00:55
 */
public class RowData implements IsSerializable {
    /**
     * Id of the records
     * If id is null it means that this row present a number of column values, but not the record in db.
     */
    private Long id;
    private String[] columns;

    public RowData(Long id, String[] columns) {
        this.id = id;
        this.columns = columns;
    }

    public RowData() {
        columns = new String[0];
    }

    /**
     * If id is null it means that this row present a number of column values, 
     * but not the record in db.
     * @return Id of the records
     */
    public Long getId() {
        return id;
    }

    public void setId(Long id) {
        this.id = id;
    }

    public String[] getColumns() {
        return columns;
    }

    public void setColumns(String[] columns) {
        if(columns != null) {
            this.columns = columns;
        }
    }

    public void copyFrom(RowData data) {
        copy(data, this);
    }

    /**
     * copy r1 to r2. both objects can be changed separately.
     * @param src first instance
     * @param dest second instance
     */
    public static void copy(RowData src, RowData dest) {
        if(src == null || dest == null) {
            throw new IllegalArgumentException("Both objects could't be a null");
        }
        dest.id = (src.id != null) ? new Long(src.id.longValue()) : null;
        String[] copy = new String[src.columns.length];
        for(int i = 0; i < copy.length; i++) {
            copy[i] = src.columns[i];
        }
        dest.columns = copy;
    }

    public RowData cloneData() {
        RowData cloned = new RowData();
        copy(cloned, this);
        return cloned;
    }

    public boolean equalsToRow(RowData row) {
        boolean ret = false;
        if(row != null) {
            if(row.columns.length == columns.length
                    && (row.id != null) ? row.id.equals(id) : id == null) {
                ret = true;
                for(int i = 0; i < columns.length; i++) {
                    if(!StringUtil.isStringsEqualsIgnoreNulls(columns[i],
                            row.columns[i])) {

                        ret = false;
                        break;
                    }
                }
            }
        }
        return ret;
    }

    public String toString() {
        return "RowData{" +
                "id='" + id + '\'' +
                ", columns=" + (columns == null ? null:Arrays.asList(columns)) +
                '}';
    }
}
