/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.config.utils;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.jxb.entityschema.Entityschema;
import com.queplix.core.jxb.entityschema.Entityschemas;
import com.queplix.core.jxb.entityschema.Property;
import com.queplix.core.utils.dao.AbstractPropertyFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

/**
 * Entity Schema factory.
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.2 $ $Date: 2005/10/26 14:03:51 $
 */

public final class EntitySchemaFactory
    extends AbstractPropertyFactory {

    // =============================================================== Constants

    // Single instance.
    private static final EntitySchemaFactory o = new EntitySchemaFactory();
    // Config file name.
    private static final String CONFIG_FILE = "entityschema.xml";

    // =============================================================== Fields

    private String defaultName;
    private Map map = new HashMap();

    // =============================================================== Constructor

    private EntitySchemaFactory() {
        try {
            // Init EntitySchema XML config.
            Entityschemas entityschemas = ( Entityschemas ) loadSysPropertiesAsObject( CONFIG_FILE, Entityschemas.class );

            // Set default schema name.
            defaultName = entityschemas.getDefault();
            if( defaultName == null ) {
                throw new GenericSystemException( "Default schema cannot be NULL!" );
            }

            // Init map of EntitySchema objects.
            for( int i = 0; i < entityschemas.getEntityschemaCount(); i++ ) {
                Entityschema entityschema = entityschemas.getEntityschema( i );
                String name = entityschema.getName();
                String className = entityschema.getEntityschemaclass();
                Property[] properties = entityschema.getProperty();

                INFO( "Try to load EntitySchema: " + name + ", class: " + className );

                // .. create new object
                EntitySchema entitySchema = ( EntitySchema ) Class.forName( className ).newInstance();

                // .. set property
                if( properties != null ) {
                    entitySchema.init( name, properties );
                }

                // .. remmeber reference
                map.put( name, entitySchema );
            }

        } catch( Exception e ) {
            ERROR( e );
            throw new GenericSystemException( e );
        }
    }

    // =============================================================== Public methods

    //
    // Get instance of EntitySchema by the name.
    //
    public EntitySchema get( String name ) {

        if( name == null ) {
            return get();
        }

        EntitySchema entitySchema = ( EntitySchema ) map.get( name );
        if( entitySchema == null ) {
            throw new GenericSystemException( "Cannot get EntitySchema '" + name + "'" );
        }
        return entitySchema;
    }

    //
    // Get default instance of EntitySchema.
    //
    public EntitySchema get() {
        return get( defaultName );
    }

    //
    // Get EntitySchema's names.
    //
    public Set names() {
        return map.keySet();
    }

    // =============================================================== Static methods

    /**
     * Get single instance
     * @return EntitySchemaFactory object
     */
    public static EntitySchemaFactory getInstance() {
        return o;
    }
}
