/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.eql;

import java.util.ArrayList;
import java.util.List;

/**
 * <p>EQL Response Record object</p>
 * @author Baranov Andrey [ALB]
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:21 $
 */

public class EQLResRecord
    implements java.io.Serializable {

    private EQLResCell[] data;
    private List eqlDResList;

    private boolean doInsert = false;
    private boolean doUpdate = false;
    private boolean doDelete = false;

    /**
     * Constructor
     * @param isNew is new record flag
     * @param size cty of cells
     */
    public EQLResRecord( boolean isNew, int size ) {

        data = new EQLResCell[size];

        unmark();
        if( isNew ) {
            markAsInsert();
        }
    }

    //
    // Special builder methods
    //

    /**
     * Add cell to the special position <code>index</code>.
     * @param resCell EQLResCell
     * @param index int
     */
    public void addData( EQLResCell resCell, int index ) {
        data[index] = resCell;
    }

    /**
     * Insert new cell to the end.
     * @param resCell EQLResCell
     */
    public void insertData( EQLResCell resCell ) {
        int length = data.length;
        EQLResCell[] __data = new EQLResCell[length + 1];
        System.arraycopy( data, 0, __data, 0, length );
        data = __data;
        addData( resCell, length );
    }

    /**
     * Add dataset response.
     * @param dRes EQLDRes
     */
    public void addDRes( EQLDRes dRes ) {
        if( eqlDResList == null ) {
            eqlDResList = new ArrayList();
        }
        eqlDResList.add( dRes );
    }

    /**
     * Get cels count.
     * @return int
     */
    public int size() {
        return data.length;
    }

    /**
     * Get dataset responses count.
     * @return int
     */
    public int getDResSize() {
        if( eqlDResList != null ) {
            return eqlDResList.size();
        } else {
            return 0;
        }
    }

    /**
     * Get EQLObject by position <code>pos</code>.
     * @param pos int
     * @return EQLObject
     */
    public EQLObject getEQLObject( int pos ) {
        return getResCell( pos ).getEQLObject();
    }

    /**
     * Get listref EQLObject by position <code>pos</code>.
     * @param pos int
     * @return EQLObject
     */
    public EQLObject getEQLListObject( int pos ) {
        EQLResCell listResCell = getResCell( pos ).getListField();
        if( listResCell != null ) {
            return listResCell.getEQLObject();
        } else {
            return null;
        }
    }

    /**
     * Get cell by position <code>pos</code>.
     * @param pos int
     * @return EQLResCell
     */
    public EQLResCell getResCell( int pos ) {
        return data[pos];
    }

    /**
     * Get cell by field <code>reqField</code>.
     * @param reqField EQLReqField
     * @return EQLResCell
     */
    public EQLResCell getResCell( EQLReqField reqField ) {
        return getResCell( reqField, 0 );
    }

    /**
     * Get cell by field <code>reqField</code> and starts from <code>startPos</code>.
     * @param reqField EQLReqField
     * @param startPos int
     * @return EQLResCell
     */
    public EQLResCell getResCell( EQLReqField reqField, int startPos ) {
        for( int i = startPos; i < data.length; i++ ) {
            EQLResCell resCell = data[i];
            // Check if resCell is not NULL and compare EQLReqField objects
            if( resCell != null && resCell.getReqField().equals( reqField ) ) {
                return resCell;
            }
        }
        return null;
    }

    /**
     * Get dataset response by position <code>pos</code>.
     * @param pos int
     * @return EQLDRes
     */
    public EQLDRes getDRes( int pos ) {
        if( eqlDResList == null ) {
            throw new IndexOutOfBoundsException( "List is empty" );
        }
        return( EQLDRes ) eqlDResList.get( pos );
    }

    /**
     * Get dataset response by datatset <code>reqDataset</code>.
     * @param reqDataset EQLReqDataset
     * @return EQLDRes
     */
    public EQLDRes getDRes( EQLReqDataset reqDataset ) {
        for( int i = 0; i < getDResSize(); i++ ) {
            EQLDRes dRes = ( EQLDRes ) eqlDResList.get( i );
            if( dRes.getReqDataset().equals( reqDataset ) ) {
                return dRes;
            }
        }
        return null;
    }

    /*
     * No javadoc
     * @see Object#hashCode
     */
    public int hashCode() {
        return data.hashCode() | eqlDResList.hashCode();
    }

    /*
     * No javadoc
     * @see Object#equals
     */
    public boolean equals( Object o ) {
        if( o == null || ! ( o instanceof EQLResRecord ) ) {
            return false;
        }

        if( o == this ) {
            return true;
        }

        EQLResRecord obj = ( EQLResRecord ) o;
        return data.equals( obj.data ) && eqlDResList.equals( obj.eqlDResList );
    }

    /*
     * No javadoc
     * @see Object#toString
     */
    public String toString() {
        return "data: " + data + "\nDRes list: [" + eqlDResList + "]";
    }

    //
    // Set current record status
    //

    protected void unmark() {
        doDelete = doInsert = doUpdate = false;
    }

    public void markAsDelete() {
        doDelete = true;
        doInsert = doUpdate = false;

        // delete all datasets too
        for( int i = 0; i < getDResSize(); i++ ) {
            ( ( EQLDRes ) eqlDResList.get( i ) ).markAllAsDelete();
        }
    }

    public void markAsUpdate() {
        doUpdate = true;
        doInsert = doDelete = false;

        // update all datasets too
        for( int i = 0; i < getDResSize(); i++ ) {
            ( ( EQLDRes ) eqlDResList.get( i ) ).markAllAsUpdate();
        }
    }

    public void markAsInsert() {
        doInsert = true;
        doUpdate = doDelete = false;

        // insert all datasets too
        for( int i = 0; i < getDResSize(); i++ ) {
            ( ( EQLDRes ) eqlDResList.get( i ) ).markAllAsInsert();
        }
    }

    //
    // Get current record status
    //

    public boolean isChanged() {

        // check flags
        if( doUpdate || doInsert ) {
            return true;
        }

        // check fields
        for( int i = 0; i < size(); i++ ) {
            if( getResCell( i ).isValueChanged() ) {
                return true;
            }
        }

        // check datasets
        for( int i = 0; i < getDResSize(); i++ ) {
            if( ( ( EQLDRes ) eqlDResList.get( i ) ).isChanged() ) {
                return true;
            }
        }

        return false;
    }

    //
    // Reset record status
    //
    public void resetStatus() {
        unmark();
    }

    //
    // Record is new?
    //
    public boolean isNew() {
        return doInsert;
    }

    //
    // Delete record?
    //
    public boolean doDelete() {
        return doDelete;
    }

    public boolean doUpdate() {
        return doUpdate;
    }

    boolean doInsert() {
        return doInsert;
    }

    public void setDoUpdate( boolean update ) {
        doUpdate = update;
    }
}
