/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.eql.utils.cache;

import com.queplix.core.jxb.entity.Entity;
import com.queplix.core.modules.eql.EQLERes;
import com.queplix.core.modules.eql.EQLRes;
import com.queplix.core.utils.log.AbstractLogger;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

/**
 * <p>EQL response cache manager.
 * <p>Stores read-only entity EQLRes objects in the Map.
 * @author Baranov Andrey [ALB]
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:33 $
 */

public class EQLResCacheManager
    extends AbstractLogger {

// -------------------- CONSTANTS --------------------------

    // Initial cache size
    private static final int INITIAL_SIZE = 500;

// -------------------- VARIABLES --------------------------

    // Cache of EQLResCacheObject objects
    private final Map cacheTab = Collections.synchronizedMap( new HashMap( INITIAL_SIZE ) );

    // Singleton
    private static EQLResCacheManager instance = new EQLResCacheManager();

// ------------------ PUBLIC METHODS -----------------------

    /**
     * Get instance of cache manager
     * @return instance of cache manager
     */
    public static EQLResCacheManager getInstance() {
        return instance;
    }

    /**
     * Get EQLResCacheObject object for given <code>entity</code>
     * @param entity given Entity
     * @return EQLResCacheObject object
     */
    public EQLResCacheObject getCacheObject( Entity entity ) {

        String cacheName = entity.getName();
        if( getLogger().isDebugEnabled() ) {
            DEBUG( "Try to find response in cache '" + cacheName + "'..." );
        }

        EQLResCacheObject co = ( EQLResCacheObject ) cacheTab.get( cacheName );
        if( co != null ) {
            // check entity versoinId attribute
            long prevVersionId = co.getVersion();
            long newVersionId = entity.getVersionId().longValue();

            if( prevVersionId != newVersionId ) {
                reset( cacheName );
                return null;
            }
        }

        return co;
    }

    /**
     * Try to find EQL response in cache
     * @param entity Entity object
     * @return EQLRes object
     */
    public EQLRes findEQLResInCache( Entity entity ) {
        EQLResCacheObject co = getCacheObject( entity );
        if( co != null ) {
            return co.getEQLRes();
        } else {
            return null;
        }
    }

    /**
     * Save entity EQL response in cache
     * @param res EQLERes object
     */
    public synchronized void saveEQLResInCache( EQLERes res ) {

        Entity entity = res.getEntity();
        String cacheName = entity.getName();
        long versionId = entity.getVersionId().longValue();

        if( getLogger().isDebugEnabled() ) {
            DEBUG( "Try to save response in cache '" + cacheName + "'..." );
        }

        EQLResCacheObject co = new EQLResCacheObject( cacheName, versionId );
        co.setEQLRes( res );
        cacheTab.put( cacheName, co );
    }

    /**
     * Reset EQLResCacheObject by the name
     * @param cacheName cache object name
     */
    public synchronized void reset( String cacheName ) {
        if( getLogger().isDebugEnabled() ) {
            DEBUG( "Remove cache object '" + cacheName + "' from cache..." );
        }
        cacheTab.remove( cacheName );
    }

// ------------------ PRIVATE METHODS ----------------------

    /**
     * Constructor
     */
    private EQLResCacheManager() {
        DEBUG( "EQLRes cache manager created: " + hashCode() );
    }

}
