/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.modules.inbox.utils.log;

import com.queplix.core.modules.inbox.InboxMessage;
import com.queplix.core.modules.mail.MailAddress;
import com.queplix.core.modules.services.utils.log.AbstractLogRecord;
import com.queplix.core.modules.services.utils.log.impl.BaseLogHandler;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Types;

/**
 * Email Service LOG handler.
 * @author Konstantin Mironov
 * @since 8 Dec 2006
 */
public class EmailLogHandler extends BaseLogHandler {

    // ----------------------------------------------------- Constants

    /** SQL to add a record to the log table. */
    public static final String ADD_EMAIL_SERVICE_LOG_SQL =
        "INSERT INTO QX_EMAIL_SERVICE_LOG (" +
        " PKEY," +
        " ACCOUNT_ID," +
        " MESSAGE_ID," +
        " E_TO," +
        " E_FROM," +
        " E_SUBJECT," +
        " OBJECT_TYPE," +
        " OBJECT_ID," +
        " OWNER_ID," +
        " WORKGROUP_ID," +
        " FILTER_ID," +
        " ROUTING_ID " +
        ") VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    // ----------------------------------------------------- Public methods

    /*
     * No javadoc
     * @see AbstractLogHandler#doInsert
     */
    public long doInsert(Connection con, AbstractLogRecord record) throws SQLException {

        // Initialization
        Long accountId = null;
        Long messageId = null;
        MailAddress[] to = null;
        MailAddress from = null;
        String subject = null;
        Long objectId = null;
        Integer objectType = null;
        Long ownerId = null;
        Long workgroupId = null;
        Long filterId = null;
        Long forwardId = null;

        AbstractInboxLogRecord lr = (AbstractInboxLogRecord)record;
        InboxMessage im = lr.getInboxMailMessage();
        if (im != null) {
            accountId = im.getAccountId();
            messageId = im.getMessageId();
            to = im.getTo();
            from = im.getFrom();
            subject = im.getSubject();
            objectType = im.getObjectType();
            objectId = im.getObjectId();
            ownerId = im.getResipient();
            workgroupId = im.getWorkgroupId();
            filterId = im.getFilterId();
            forwardId = im.getRoutingRuleId();

        } else if (lr.getAccountId() != null) {
            accountId = lr.getAccountId();
        } // if (im != null)

        // Insert master log record
        long pkey = super.doInsert(con, lr);

        // Insert new record in the QX_EMAIL_SERVICE_LOG table
        PreparedStatement stat = null;
        try {
            stat = sqlWrapper.doPreparedStatement(con, ADD_EMAIL_SERVICE_LOG_SQL);
            stat.setLong(1, pkey);

            if (accountId != null) {
                stat.setLong(2, accountId.longValue());
            } else {
                stat.setNull(2, Types.NUMERIC);
            }

            if (messageId != null) {
                stat.setLong(3, messageId.longValue());
            } else {
                stat.setNull(3, Types.NUMERIC);
            }

            if (to != null) {
                stat.setString(4, MailAddress.toRfcString(to));
            } else {
                stat.setNull(4, Types.VARCHAR);
            }

            if (from != null) {
                stat.setString(5, from.toRfcString());
            } else {
                stat.setNull(5, Types.VARCHAR);
            }

            stat.setString(6, subject);

            if (objectType != null) {
                stat.setInt(8, objectType.intValue());
            } else {
                stat.setNull(8, Types.NUMERIC);
            }

            if (objectId != null) {
                stat.setLong(7, objectId.longValue());
            } else {
                stat.setNull(7, Types.NUMERIC);
            }

            if (ownerId != null) {
                stat.setLong(9, ownerId.longValue());
            } else {
                stat.setNull(9, Types.NUMERIC);
            }

            if (workgroupId != null) {
                stat.setLong(10, workgroupId.longValue());
            } else {
                stat.setNull(10, Types.NUMERIC);
            }

            if (filterId != null) {
                stat.setLong(11, filterId.longValue());
            } else {
                stat.setNull(11, Types.NUMERIC);
            }

            if (forwardId != null) {
                stat.setLong(12, forwardId.longValue());
            } else {
                stat.setNull(12, Types.NUMERIC);
            }

            sqlWrapper.executeUpdate(stat);

        } finally {
            sqlWrapper.closeConnection(stat);
        } // try

        return pkey;
    } // doInsert(Connection, AbstractLogRecord)
} // class EmailLogHandler 
