/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.services;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.modules.services.ejb.ScriptManager;
import com.queplix.core.modules.services.ejb.ScriptManagerHome;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.cache.CacheObjectManager;
import com.queplix.core.utils.sql.SqlWrapper;
import com.queplix.core.utils.sql.SqlWrapperFactory;

import java.io.Serializable;
import java.sql.Connection;
import java.util.HashMap;
import java.util.Map;

/**
 * Action context. Contains all action data.
 *
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:58 $
 */
public class ActionContext
    implements Serializable {

    // ================================================================== Fields

    private LogonSession ls;
    private Map params = new HashMap();

    // SQL wrapper.
    private final SqlWrapper sqlWrapper = SqlWrapperFactory.getSqlWrapper();

    // EJB cache manager.
    private final CacheObjectManager com = new CacheObjectManager();

    // ========================================================== Initialization

    public ActionContext( LogonSession ls ) {
        this.ls = ls;
    }

    // =================================================== Fields access methods

    /**
     * User logon session getter.
     * @return user logon session
     */
    public final LogonSession getLogonSession() {
        return ls;
    }

    /**
     * SQL wrapper getter.
     * @return SQL wrapper
     */
    public SqlWrapper getSqlWrapper() {
        return sqlWrapper;
    }

    /**
     * Database connection getter.
     * @return connection object
     */
    public Connection getConnection() {
        return getSqlWrapper().doConnection();
    }

    /**
     * Cache Object Manager reference getter.
     * @return EJB cache manager reference
     */
    public final CacheObjectManager getCOM() {
        return com;
    }

    /**
     * ScriptManager EJB reference getter.
     * @return EJB remote interface
     */
    public final ScriptManager getScriptManager() {
        return( ScriptManager ) com.getRemoteObject( JNDINames.ScriptManagerRemote, ScriptManagerHome.class );
    }

    // =============================================== Parameters access methods

    /**
     * Action parameters getter.
     * @return parameters map
     */
    public final Map getParameters() {
        return params;
    }

    /**
     * Action paramaters setter.
     * @param parameters map of parameters
     */
    public final void addParameters( Map parameters ) {
        params.putAll( parameters );
    }

    /**
     * Action parameter getter.
     *
     * @param name parameter name
     * @return parameter value as object, or <b>null</b>, if not found
     */
    public final Object getParameter( String name ) {
        return params.get( name );
    }

    /**
     * Action parameter setter.
     *
     * @param name parameter name
     * @param value parameter value
     */
    public final void addParameter( String name, Object value ) {
        params.put( name, value );
    }

    /**
     * String action parameter getter.
     *
     * @param name parameter name
     * @return parameter value as string, or <b>null</b>, if not found
     */
    public final String getParamAsString( String name ) {
        Object o = getParameter( name );
        return( o == null ) ? null : o.toString();
    }

    /**
     * Integer action parameter getter.
     *
     * @param name parameter name
     * @return parameter value as integer, or <b>null</b>, if not found
     */
    public final Integer getParamAsInt( String name ) {

        Object o = getParameter( name );
        if( o == null ) {
            return null;
        } else if( o instanceof Integer ) {
            return( Integer ) o;
        }

        try {
            return new Integer( o.toString() );
        } catch( NumberFormatException ex ) {
            throw new IllegalArgumentException( "Non-numeric parameter '" + name + "'" );
        }

    }

    /**
     * Long integer action parameter getter.
     *
     * @param name parameter name
     * @return parameter value as long integer, or <b>null</b>, if not found
     */
    public final Long getParamAsLong( String name ) {

        Object o = getParameter( name );
        if( o == null ) {
            return null;
        } else if( o instanceof Long ) {
            return( Long ) o;
        }

        try {
            return new Long( o.toString() );
        } catch( NumberFormatException ex ) {
            throw new IllegalArgumentException( "Non-numeric parameter '" + name + "'" );
        }

    }

    /**
     * Boolean action parameter getter.
     *
     * @param name parameter name
     * @return parameter value as boolean, or <b>null</b>, if not found
     */
    public final Boolean getParamAsBoolean( String name ) {
        Object o = getParameter( name );
        if( o == null ) {
            return null;
        } else if( o instanceof Boolean ) {
            return( Boolean ) o;
        }
        return new Boolean( o.toString() );
    }

    // ======================================================= Inherited methods

    /* (non-Javadoc)
     * @see java.lang.Object#toString()
     */
    public String toString() {
        return "[logon session = " + ls + ", parameters = " + params + "]";
    }

}
