/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.modules.services.utils;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.modules.services.jxb.ServicesConfig;
import com.queplix.core.utils.cache.Cache;
import com.queplix.core.utils.cache.CacheFactory;
import com.queplix.core.utils.dao.AbstractPropertyFactory;

/**
 * Services property access factory
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.2 $ $Date: 2005/10/26 14:03:51 $
 */

public final class ServicesPropertyFactory
    extends AbstractPropertyFactory {

    // ================================================================= Constants

    // Single instance.
    private static final ServicesPropertyFactory o = new ServicesPropertyFactory();

    // Config file name.
    private static final String CONFIG_FILE = "services.xml";

    // ================================================================= Fields

    // XML services config
    private ServicesConfig servicesConfig;

    // DAO cached class.
    private ScriptDAO scriptDAO;
    private TaskDAO taskDAO;

    // Cache cached classes.
    private Cache scriptCache;
    private Cache taskCache;

    // ===================================================== Constructor

    // Constructor - blocks direct instantiation.
    private ServicesPropertyFactory() {

        // XML config initialization
        servicesConfig = ( ServicesConfig ) loadSysPropertiesAsObject( CONFIG_FILE, ServicesConfig.class );

        //
        // DAO initialization
        //

        scriptDAO = ( ScriptDAO ) initObject( servicesConfig.getScriptConfig().getDao().getClassName(),
                                              servicesConfig.getScriptConfig().getDao().getParam() );

        taskDAO = ( TaskDAO ) initObject( servicesConfig.getTaskConfig().getDao().getClassName(),
                                          servicesConfig.getTaskConfig().getDao().getParam() );

        //
        // Cache initialization
        //

        scriptCache = ( Cache ) CacheFactory.getInstance().initObject(
            servicesConfig.getScriptConfig().getCache().getClassName(),
            servicesConfig.getScriptConfig().getCache().getParam() );

        taskCache = ( Cache ) CacheFactory.getInstance().initObject(
            servicesConfig.getTaskConfig().getCache().getClassName(),
            servicesConfig.getTaskConfig().getCache().getParam() );
    }

    // ===================================================== Static methods

    /**
     * Get single instance
     * @return ServicesPropertyFactory object
     */
    public static ServicesPropertyFactory getInstance() {
        return o;
    }

    // ===================================================== Public methods

    /**
     * Instantiates the script DAO
     * @return script DAO
     */
    public ScriptDAO getScriptDAO() {
        if( scriptDAO == null ) {
            throw new GenericSystemException( "Can't get script DAO" );
        }

        return scriptDAO;
    }

    /**
     * Instantiates the task DAO
     * @return task DAO
     */
    public TaskDAO getTaskDAO() {
        if( taskDAO == null ) {
            throw new GenericSystemException( "Can't get task DAO" );
        }

        return taskDAO;
    }

    /**
     * Instantiates the script cache
     * @return cache
     */
    public Cache getScriptCache() {
        if( scriptCache == null ) {
            throw new GenericSystemException( "Can't get script cache" );
        }
        return scriptCache;
    }

    /**
     * Instantiates the task cache
     * @return cache
     */
    public Cache getTaskCache() {
        if( taskCache == null ) {
            throw new GenericSystemException( "Can't get task cache" );
        }
        return taskCache;
    }
}
