/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.utils.ejb;

import com.queplix.core.error.ErrorHelper;
import com.queplix.core.error.GenericSystemException;
import com.queplix.core.utils.cache.CacheObjectManager;
import com.queplix.core.utils.log.AbstractLogger;
import com.queplix.core.utils.sql.SqlWrapper;
import com.queplix.core.utils.sql.SqlWrapperFactory;

import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

/**
 * Facade for all session EJBs.
 * @author [ALB] Baranov Andrey
 * @author [ONZ] Oleg N. Zhovtanyuk
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:31:19 $
 */

public abstract class AbstractSessionEJB
    extends AbstractLogger
    implements SessionBean {

    // ------------------------------------------------------- Fields

    /** Session EJB context. */
    protected SessionContext sessionContext;

    // Sql wrapper instance.
    private SqlWrapper sqlWrapper;

    // Cache object manager
    private final CacheObjectManager com = new CacheObjectManager();

    // ------------------------------------------------------- Common session EJB implementation

    /**
     * Remove bean
     */
    public void ejbRemove() {
        INFO( "EJB remove - " + hashCode() );
    }

    /**
     * Activate bean
     */
    public void ejbActivate() {}

    /**
     * Passivate bean
     */
    public void ejbPassivate() {}

    /**
     * Set Session context
     * @param sessionContext Session context
     */
    public void setSessionContext( SessionContext sessionContext ) {
        INFO( "EJB set session context - " + hashCode() );

        this.sessionContext = sessionContext;
        this.sqlWrapper = SqlWrapperFactory.getSqlWrapper();
    }

    // ------------------------------------------------------- New features

    /**
     * Rolls back a transaction.
     */
    public void setRollbackOnly() {
        try {
            sessionContext.setRollbackOnly();
        } catch( IllegalStateException ex ) {
            WARN( "Can`t rollback transaction. " + ex.getMessage() );
        }
    }

    /**
     * Gets {@link SqlWrapper SqlWrapper} implementation reference.
     * @return SqlWrapper object
     */
    public SqlWrapper getSqlWrapper() {
        return sqlWrapper;
    }

    /**
     * Gets EJB home reference.
     * @param jndiName JNDI name
     * @param homeClass home class
     * @return remote home object
     */
    public Object getRemoteHome( String jndiName, Class homeClass ) {
        try {
            return com.getRemoteHome( jndiName, homeClass );
        } catch( Exception ex ) {
            throwException( "Can't get remote home '" + jndiName + "'", ex );
            return null;
        }
    }

    /**
     * Gets EJB local home reference.
     * @param jndiName JNDI name
     * @param homeClass home class
     * @return local home object
     */
    public Object getLocalHome( String jndiName, Class homeClass ) {

        try {
            return com.getLocalHome( jndiName, homeClass );
        } catch( Exception ex ) {
            throwException( "Can't get local home '" + jndiName + "'", ex );
            return null;
        }
    }

    /**
     * Gets EJB reference.
     * @param jndiName JNDI name
     * @param homeClass home class
     * @return remote object
     */
    public Object getRemoteObject( String jndiName, Class homeClass ) {
        try {
            return com.getRemoteObject( jndiName, homeClass );
        } catch( Exception ex ) {
            throwException( "Can't get remote object '" + jndiName + "'", ex );
            return null;
        }
    }

    /**
     * Gets EJB local reference.
     * @param jndiName JNDI name
     * @param homeClass home class
     * @return local object
     */
    public Object getLocalObject( String jndiName, Class homeClass ) {
        try {
            return com.getLocalObject( jndiName, homeClass );
        } catch( Exception ex ) {
            throwException( "Can't get local object '" + jndiName + "'", ex );
            return null;
        }
    }

    /**
     * Throws the {@link GenericSystemException GenericSystemException}.
     * @param err error message
     */
    public void throwException( String err ) {
        throwException( err, null );
    }

    /**
     * Throws the {@link GenericSystemException GenericSystemException}.
     * @param t exception object
     */
    public void throwException( Throwable t ) {
        throwException( "Caught in " + getClass() + ": " + t.getMessage(), t );
    }

    /**
     * Throws the {@link GenericSystemException GenericSystemException}.
     * @param err error message
     * @param t exception object
     */
    public void throwException( String err, Throwable t ) {
        ERROR( "[!!!] Exception in EJB: " + getClass().getName() );
        ErrorHelper.throwSystemException( err, t, this );
        ERROR( "[!!!]" );
    }

} // end of class
