/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.common.ui;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.EventPreview;
import com.google.gwt.user.client.ui.KeyboardListenerCollection;
import com.google.gwt.user.client.ui.PopupListener;
import com.google.gwt.user.client.ui.RootPanel;
import com.google.gwt.user.client.ui.SimplePanel;
import com.google.gwt.user.client.ui.SourcesPopupEvents;
import com.google.gwt.user.client.ui.impl.PopupImpl;

/**
 * Fork of GWT PopupPanel: the non-modal popup
 * All the original code belongs to Google Inc.
 *
 * @see com.google.gwt.user.client.ui.PopupPanel
 * @author: Vasily Mikhailitchenko
 * @since: 26 Oct 2006
 */
public class NonModalPopup extends SimplePanel implements SourcesPopupEvents, EventPreview {

  private static PopupImpl impl = (PopupImpl) GWT.create(PopupImpl.class);

  private NonModalPopupListenerCollection popupListeners;
  private boolean showing, autoHide;

  /**
   * Creates an empty popup panel. A child widget must be added to it before it
   * is shown.
   */
  public NonModalPopup() {
    super(impl.createElement());
  }

  /**
   * Creates an empty popup panel, specifying its "auto-hide" property.
   *
   * @param autoHide <code>true</code> if the popup should be automatically
   *          hidden when the user clicks outside of it
   */
  public NonModalPopup(boolean autoHide) {
    this();
    this.autoHide = autoHide;
  }

  public void addPopupListener(PopupListener listener) {
    if (popupListeners == null)
      popupListeners = new NonModalPopupListenerCollection();
    popupListeners.add(listener);
  }

  /**
   * Hides the popup. This has no effect if it is not currently visible.
   */
  public void hide() {
    hide(false);
  }

  public boolean onEventPreview(Event event) {
    int type = DOM.eventGetType(event);
    switch (type) {
      case Event.ONKEYDOWN: {
        return onKeyDownPreview((char)DOM.eventGetKeyCode(event),
          KeyboardListenerCollection.getKeyboardModifiers(event));
      }
      case Event.ONKEYUP: {
        return onKeyUpPreview((char)DOM.eventGetKeyCode(event),
          KeyboardListenerCollection.getKeyboardModifiers(event));
      }
      case Event.ONKEYPRESS: {
        return onKeyPressPreview((char)DOM.eventGetKeyCode(event),
          KeyboardListenerCollection.getKeyboardModifiers(event));
      }

      case Event.ONMOUSEDOWN:
      case Event.ONMOUSEUP:
      case Event.ONMOUSEMOVE:
      case Event.ONCLICK:
      case Event.ONDBLCLICK: {
        break;
      }
    }
    return true;
  }

  /**
   * Popups get an opportunity to preview keyboard events before they are passed
   * to any other widget.
   *
   * @param key the key code of the depressed key
   * @param modifiers keyboard modifiers, as specified in
   *          {@link com.google.gwt.user.client.ui.KeyboardListener}.
   * @return <code>false</code> to suppress the event
   */
  public boolean onKeyDownPreview(char key, int modifiers) {
    return true;
  }

  /**
   * Popups get an opportunity to preview keyboard events before they are passed
   * to any other widget.
   *
   * @param key the unicode character pressed
   * @param modifiers keyboard modifiers, as specified in
   *          {@link com.google.gwt.user.client.ui.KeyboardListener}.
   * @return <code>false</code> to suppress the event
   */
  public boolean onKeyPressPreview(char key, int modifiers) {
    return true;
  }

  /**
   * Popups get an opportunity to preview keyboard events before they are passed
   * to any other widget.
   *
   * @param key the key code of the released key
   * @param modifiers keyboard modifiers, as specified in
   *          {@link com.google.gwt.user.client.ui.KeyboardListener}.
   * @return <code>false</code> to suppress the event
   */
  public boolean onKeyUpPreview(char key, int modifiers) {
    return true;
  }

  public void removePopupListener(PopupListener listener) {
    if (popupListeners != null)
      popupListeners.remove(listener);
  }

  public void setPixelSize(int width, int height) {
    if (width >= 0)
      setWidth("" + width);
    if (height >= 0)
      setHeight("" + height);
  }

  /**
   * Sets the popup's position relative to the browser's client area.
   *
   * @param left the left position, in pixels
   * @param top the top position, in pixels
   */
  public void setPopupPosition(int left, int top) {
    RootPanel.get().setWidgetPosition(this, left, top);
  }


  /**
   * Shows the popup. It must have a child widget before this method is called.
   */
  public void show() {
    if (showing)
      return;
    showing = true;
    DOM.addEventPreview(this);

    RootPanel.get().add(this);
  }

  private void hide(boolean autoClosed) {
    if (!showing)
      return;
    showing = false;
    DOM.removeEventPreview(this);

    RootPanel.get().remove(this);
    if (popupListeners != null)
      popupListeners.firePopupClosed(this, autoClosed);
  }

} 
