/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.common.ui;

import com.google.gwt.user.client.Event;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.ClickListener;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.FocusListenerAdapter;
import com.google.gwt.user.client.ui.FocusPanel;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.HorizontalPanel;
import com.google.gwt.user.client.ui.KeyboardListenerAdapter;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.MouseListenerAdapter;
import com.google.gwt.user.client.ui.SourcesTabEvents;
import com.google.gwt.user.client.ui.TabListener;
import com.google.gwt.user.client.ui.TabListenerCollection;
import com.google.gwt.user.client.ui.Widget;

/**
 * Fork of GWT TabBar with enhanced functionality
 * All the original code belongs to Google Inc.
 *
 * @see com.google.gwt.user.client.ui.TabBar
 * @author Vasily Mikhailitchenko
 * @since 14.11.2006
 * @noinspection ALL
 */
public class QTabBar extends Composite implements SourcesTabEvents, SourcesTabMouseEvents, ClickListener{
    public static final int KEY_SPACE = 32;

    private HorizontalPanel panel = new HorizontalPanel();
    private Widget selectedTab;
    private Widget highlightedTab;
    private TabListenerCollection tabListeners;
    private TabMouseListenerCollection tabMouseListeners;

    private MouseListenerAdapter mouseAdapter;
    private FocusListenerAdapter focusAdapter;
    private KeyboardListenerAdapter keyListener;

    private boolean isInTabPabel = false;

    public QTabBar() {
        initWidget(panel);
        sinkEvents(Event.ONCLICK);
        setStyleName("gwt-TabBar");

        panel.setVerticalAlignment(HorizontalPanel.ALIGN_BOTTOM);

        HTML first = new HTML("&nbsp;", true), rest = new HTML("&nbsp;", true);
        first.setStyleName("gwt-TabBarFirst");
        rest.setStyleName("gwt-TabBarRest");
        first.setHeight("100%");
        rest.setHeight("100%");

        panel.add(first);
        panel.add(rest);
        first.setHeight("100%");
        panel.setCellHeight(first, "100%");
        panel.setCellWidth(rest, "100%");
        
        initMouseAdapter();
        initFocusAdapter();
        initKeyListener();
    }

    public void addTab(String text) {
        insertTab(text, getTabCount());
    }

    public void addTab(String text, boolean asHTML) {
        insertTab(text, asHTML, getTabCount());
    }

    public int getSelectedTab() {
        if (selectedTab == null){
            return -1;
        }
        return panel.getWidgetIndex(selectedTab) - 1;
    }

    public int getTabCount() {
        return panel.getWidgetCount() - 2;
    }


    public String getTabHTML(int index) {
        if (index >= getTabCount()) {
            return null;
        }
        Widget widget = panel.getWidget(index + 1);
        if (widget instanceof HTML) {
            return ((HTML) widget).getHTML();
        } else {
            return ((Label) widget).getText();
        }
    }

    public void insertTab(String text, boolean asHTML, int beforeIndex) {
        if ((beforeIndex < 0) || (beforeIndex > getTabCount()))
          throw new IndexOutOfBoundsException();

        Label title;
        if (asHTML) {
            title = new HTML(text);
        } else {
            title = new Label(text);
        }
        
        FocusPanel item = new FocusPanel(title);
        
        title.setWordWrap(false);        
        item.addClickListener(this);
        item.addMouseListener(mouseAdapter);
        item.addFocusListener(focusAdapter);
        item.addKeyboardListener(keyListener);
        item.setStyleName("gwt-TabBarItem");
        panel.insert(item, beforeIndex + 1);
    }

    public void insertTab(String text, int beforeIndex) {
        insertTab(text, false, beforeIndex);
    }

    public Widget getTab(int index){
        if(index >= getTabCount()) {
            return null;
        }
        return panel.getWidget(index + 1);
    }

    public void onClick(Widget sender) {
        for (int i = 1; i < panel.getWidgetCount() - 1; ++i) {
          if (panel.getWidget(i) == sender) {
            selectTab(i - 1);
            return;
          }
        }
    }

    public void removeTab(int index) {
        checkTabIndex(index);

        Widget toRemove = panel.getWidget(index + 1);
        if (toRemove == selectedTab)
          selectedTab = null;
        panel.remove(toRemove);
    }

    public void addTabListener(TabListener listener) {
        if (tabListeners == null) {      
            tabListeners = new TabListenerCollection();
        }
        tabListeners.add(listener);
    }
    
    public void removeTabListener(TabListener listener) {
        if (tabListeners != null) {
            tabListeners.remove(listener);
        }
    }
    
    public void addTabMouseListener(TabMouseListener listener) {
        if(tabMouseListeners == null){
            tabMouseListeners = new TabMouseListenerCollection();
        }
        tabMouseListeners.add(listener);
    }

    public void removeTabMouseListener(TabMouseListener listener) {
        if (tabMouseListeners != null) {
            tabMouseListeners.remove(listener);
        }
    } 

    public boolean selectTab(int index) {
        checkTabIndex(index);

        if(index != getSelectedTab()) {
            if (tabListeners != null) {
              if (!tabListeners.fireBeforeTabSelected(this, index))
                return false;
            }

            setSelectionStyle(selectedTab, false);
            setHighlightStyle(selectedTab, false);
            
            if (index == -1) {
              selectedTab = null;
              return true;
            }

            selectedTab = panel.getWidget(index + 1);
            setSelectionStyle(selectedTab, true);
            setHighlightStyle(selectedTab, false);

            if (tabListeners != null)
              tabListeners.fireTabSelected(this, index);
            return true;
        } else {
            return true;
        }
    }
    
    private void highlightTab(int index){
        setHighlightStyle(highlightedTab, false);
        highlightedTab = panel.getWidget(index+1);
        if(!highlightedTab.equals(selectedTab)) {
            setHighlightStyle(highlightedTab, true);
        }
    }
    
    public void mouseOverTab(int index){
        highlightTab(index);
        if(tabMouseListeners != null){
            tabMouseListeners.fireTabMouseOver(this, index);
        }
    }
    
    public void mouseOutTab(int index){
        if(isInTabPabel){
            setHighlightStyle(highlightedTab, false);
        }
        if(tabMouseListeners != null){
            tabMouseListeners.fireTabMouseOut(this, index);
        }
    }

    public void setInTabPanel(boolean isInTabPanel){
        this.isInTabPabel = isInTabPanel;        
    }

    private void checkTabIndex(int index) {
        if ((index < -1) || (index >= getTabCount())) {
            throw new IndexOutOfBoundsException();
        }
    }
    
    private void setHighlightStyle(Widget item, boolean highlighted) {
        if((item != null)) {
            if(highlighted) {
                item.addStyleName("gwt-TabBarItem-highlighted");
            } else {
                item.removeStyleName("gwt-TabBarItem-highlighted");
            }
        }        
    }
    
    public void removeHighlight() {
        setHighlightStyle(highlightedTab, false);
    }
    
    private void setSelectionStyle(Widget item, boolean selected) {
        if (item != null) {
          if (selected)
            item.addStyleName("gwt-TabBarItem-selected");
          else
            item.removeStyleName("gwt-TabBarItem-selected");
        }
    }

    private int getWidgetIndex(Widget sender){
        for (int i = 1; i < panel.getWidgetCount() - 1; ++i) {
            if (panel.getWidget(i) == sender) {
                return i;
            }
        }
        return -1;
    }

    private void initMouseAdapter() {
        mouseAdapter = new MouseListenerAdapter(){
            public void onMouseEnter(Widget sender){
                mouseOverTab(getWidgetIndex(sender) - 1);
            }
            
            public void onMouseLeave(Widget sender) {
                mouseOutTab(getWidgetIndex(sender) - 1);
            }
        };
    }

    private void initFocusAdapter(){
        focusAdapter = new FocusListenerAdapter(){
              public void onFocus(Widget sender) {
                  int tabIndex = getWidgetIndex(sender) -1;
                  if (tabIndex != getSelectedTab()){
                      mouseOverTab(tabIndex);
                  }
              }

              public void onLostFocus(Widget sender) {
                  int tabIndex = getWidgetIndex(sender) -1;
                  if (tabIndex != getSelectedTab()){
                      mouseOutTab(tabIndex);
                  }
              }
        };
    }

    private void initKeyListener(){
        keyListener = new KeyboardListenerAdapter(){
            public void onKeyPress(Widget sender, char keyCode, int modifiers) {
                if(keyCode == KEY_SPACE){
                    selectTab(getWidgetIndex(sender) - 1);
                }
            }
        };
    }

    /**
     * Returns the size of the area from the beginning of QTabBar up to the specified tab
     * @param index of the tab
     * @return area size in pixels
     */
    public int getTabOffsetLeft(int index){
        return DOM.getIntAttribute(DOM.getParent(getTab(index).getElement()), "offsetLeft");
    }

}
