/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT 
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the 
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.common.ui.resizable;

import com.google.gwt.user.client.ui.ScrollPanel;
import com.google.gwt.user.client.ui.Widget;
import com.queplix.core.client.common.StringUtil;
import com.queplix.core.client.common.ui.*;

/**
 * Resizable scroll panel.
 * @author Sultan Tezadov
 * @since 21 Oct 2006
 */
public class ResizableScrollPanel extends ScrollPanel implements Resizable {

    /** Creates a new instance of ResizableScrollPanel */
    public ResizableScrollPanel() {
        super();
    }

    public ResizableScrollPanel(Widget child) {
        super();
        add(child);
    }
        
    public void add(Widget w) {
        if (w instanceof Resizable) {
            throw new IllegalArgumentException("Invalid argument: instance of " + 
                    "Resizable. No need to wrap it with ResizableScrollPanel.");
        }
        super.add(w);
    }

    // ======================== Resizable implementation =====================
    private int offsetHeight;
    private int offsetWidth;
    
    public void setOffsetHeight(int height) {
        if ((height < 0)  ||  (offsetHeight == height)) return; // invalid or unchanged value
        offsetHeight = height;
        if (isAttached()) {
            doSetOffsetHeight();
        }
    }

    public void setOffsetWidth(int width) {
        if ((width < 0)  ||  (offsetWidth == width)) return; // invalid or unchanged value
        offsetWidth = width;
        if (isAttached()) {
            doSetOffsetWidth();
        }
    }
    
    private void doSetOffsetHeight() {
        super.setHeight(StringUtil.pixelToSize(offsetHeight));
        int actualHeight = getOffsetHeight();
        if (actualHeight != offsetHeight) {
            // correct the height
            int delta = actualHeight - offsetHeight;
            int correctedHeight = offsetHeight - delta;
            super.setHeight(StringUtil.pixelToSize(correctedHeight));
        }
    }

    private void doSetOffsetWidth() {
        super.setWidth(StringUtil.pixelToSize(offsetWidth));
        int actualWidth = getOffsetWidth();
        if (actualWidth != offsetWidth) {
            // correct the width
            int delta = actualWidth - offsetWidth;
            int correctedWidth = offsetWidth - delta;
            super.setWidth(StringUtil.pixelToSize(correctedWidth));
        }
    }
    
    protected void onAttach() {
        super.onAttach();
        if (offsetHeight > 0) { // offsetHeight was set
            doSetOffsetHeight();
        }
        if (offsetWidth > 0) { // offsetWidth was set
            doSetOffsetWidth();
        }
    }

    public int getOffsetWidth() {
        int retValue = super.getOffsetWidth();
        retValue = (retValue != 0) ? retValue : offsetWidth;
        return retValue;
    }

    public int getOffsetHeight() {
        int retValue = super.getOffsetHeight();
        retValue = (retValue != 0) ? retValue : offsetHeight;
        return retValue;
    }

    public void setHeight(String height) {
        setOffsetHeight(StringUtil.sizeToPixel(height));
    }

    public void setWidth(String width) {
        setOffsetWidth(StringUtil.sizeToPixel(width));
    }
    // ===================== End of Resizable implementation ==================

}
