/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.core.client.controls;

import com.queplix.core.client.app.vo.CheckBoxMeta;
import com.queplix.core.client.app.vo.DateFieldMeta;
import com.queplix.core.client.app.vo.EntityLinkFieldMeta;
import com.queplix.core.client.app.vo.EntityReferenceMeta;
import com.queplix.core.client.app.vo.FieldMeta;
import com.queplix.core.client.app.vo.HistoryFieldMeta;
import com.queplix.core.client.app.vo.InFormGridFieldMeta;
import com.queplix.core.client.app.vo.ListboxFieldMeta;
import com.queplix.core.client.app.vo.MemoFieldMeta;
import com.queplix.core.client.app.vo.MultiselectFieldMeta;
import com.queplix.core.client.app.vo.TextBoxFieldMeta;
import com.queplix.core.client.app.vo.TextareaFieldMeta;
import com.queplix.core.client.controls.checkbox.QCheckBox;
import com.queplix.core.client.controls.datefield.QDateField;
import com.queplix.core.client.controls.entitylink.QEntityLink;
import com.queplix.core.client.controls.entityreference.QEntityReference;
import com.queplix.core.client.controls.form.IncorrectQFormElementData;
import com.queplix.core.client.controls.form.WidgetTypeNotSupported;
import com.queplix.core.client.controls.history.QHistory;
import com.queplix.core.client.controls.informgrid.QInformGrid;
import com.queplix.core.client.controls.listbox.QListBox;
import com.queplix.core.client.controls.memo.QMemo;
import com.queplix.core.client.controls.multiselect.QMultiSelect;
import com.queplix.core.client.controls.textarea.QTextArea;
import com.queplix.core.client.controls.textbox.QTextBox;

/**
 * Class implements the factory pattern. It encapsulates concrete classes of created entities.
 *
 * @author Sergey Kozmin
 * @since 26 Sep 2006
 */
public class QFormComponentsFactory {
    /**
     * Creates and fills up {@link com.queplix.core.client.controls.QFormElement} component according to it's {@link com.queplix.core.client.app.vo.FieldType}
     *
     * @param element      prototype element
     * @param dataListener listener to be added to appropriate components
     * @param layout       @deprecated, need to set it up on server side.
     * @return filled up {@link com.queplix.core.client.controls.QFormElement}  instance
     * @throws com.queplix.core.client.controls.form.IncorrectQFormElementData
     *          to be thrown when element has wrong data
     * @throws com.queplix.core.client.controls.form.WidgetTypeNotSupported
     *          to be thrown when element has unsupported type of data.
     */
    public static QFormElement createWidetForElement(FieldMeta element,
                                                     DataRequirementsListener dataListener,
                                                     int layout)
            throws
            WidgetTypeNotSupported,
            IncorrectQFormElementData {
        QFormElement retControl;
        if(element != null) {
            int type = element.getDataType();
            switch(type) {
                case FieldMeta.CHECKBOX: {
                    retControl = new QCheckBox((CheckBoxMeta) element, layout);
                    break;
                }
                case FieldMeta.DATEFIELD: {
                    retControl = new QDateField((DateFieldMeta) element,
                            layout);
                    break;
                }
                case FieldMeta.LISTBOX: {
                    retControl = new QListBox((ListboxFieldMeta) element,
                            layout);
                    retControl.getBaseController().setDataRequirementsListener(
                            dataListener);
                    break;
                }
                case FieldMeta.MEMO: {
                    retControl = new QMemo((MemoFieldMeta) element, layout);
                    break;
                }
                case FieldMeta.HISTORY: {
                    retControl = new QHistory((HistoryFieldMeta) element);
                    retControl.getBaseController().setDataRequirementsListener(
                            dataListener);
                    break;
                }
                case FieldMeta.MULTISELECT: {
                    retControl = new QMultiSelect(
                            (MultiselectFieldMeta) element);
                    retControl.getBaseController().setDataRequirementsListener(
                            dataListener);
                    break;
                }
                case FieldMeta.TEXTAREA: {
                    retControl = new QTextArea((TextareaFieldMeta) element,
                            layout);
                    break;
                }
                case FieldMeta.TEXTBOX: {
                    TextBoxFieldMeta fieldMeta = (TextBoxFieldMeta) element;
                    retControl = new QTextBox(fieldMeta, layout);
                    break;
                }
                case FieldMeta.IN_FORM_GRID: {
                    retControl = new QInformGrid((InFormGridFieldMeta) element);
                    retControl.getBaseController().setDataRequirementsListener(
                            dataListener);
                    break;
                }
                case FieldMeta.ENTITYREFERENCE: {
                    EntityReferenceMeta erm = (EntityReferenceMeta) element;
                    retControl = new QEntityReference(erm, layout);
                    retControl.getBaseController().setDataRequirementsListener(
                            dataListener);
                    break;
                }
                case FieldMeta.ENTITYLINK: {
                    retControl = new QEntityLink((EntityLinkFieldMeta) element);
                    break;
                }
                default: {
                    throw new WidgetTypeNotSupported("Unsupported widget type: "
                            + type + ". See QControlType interface.");
                }
            }
        } else {
            throw new IncorrectQFormElementData(
                    "Control prototype could not be a null object");
        }
        return retControl;
    }
}
