/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.controls.informgrid;

import com.queplix.core.client.app.vo.FieldDataRequest;
import com.queplix.core.client.app.vo.GetControlDataInformRequest;
import com.queplix.core.client.app.vo.GetRecordsDataInformRequest;
import com.queplix.core.client.app.vo.InFormDataRequest;
import com.queplix.core.client.app.vo.InFormGridFieldMeta;
import com.queplix.core.client.app.vo.RecordFilter;
import com.queplix.core.client.app.vo.RowData;
import com.queplix.core.client.app.vo.SearchInformRequest;
import com.queplix.core.client.common.ui.DialogHelper;
import com.queplix.core.client.controls.QFormElementControllerImpl;
import com.queplix.core.client.controls.QFormElementModel;
import com.queplix.core.client.controls.QFormElementView;

import java.util.Collection;
import java.util.List;

class QInformGridControllerImpl extends QFormElementControllerImpl 
        implements QInformGridController,
        QInformGridView.ViewActionsListener {
    private static final String INCORRECT_STATE_CHANGE = "Internal error. Unit wants to change to uncknown QForm state.";

    private QInformGridModelImpl model;
    private QInformGridViewImpl view;

    private QInformGridState searchState;
    private QInformGridState selectedState;
    private QInformGridState editState;

    private QInformGridState currentState;

    private GetControlDataInformRequest getControlDataReq;
    private GetRecordsDataInformRequest getRecordsDataReq;
    private SearchInformRequest searchReq;

    public QInformGridControllerImpl(QInformGridModelImpl model, QInformGridViewImpl view) {
        this.model = model;
        this.view = view;

        InformGridStateContext context = new InformGridStateContext();
        searchState = new SearchGridState(context);
        editState = new EditGridState(context);
        selectedState = new SelectedGridState(context);

        currentState = searchState;
        currentState.enterToState();
        view.addActionListener(this);
    }

    protected void turnToState(QInformGridState newState) {
        if (!currentState.equals(newState)) {
//            System.out.println("<QInformGridControllerImpl.turnToState.56>turn to state = " + getStateName(newState.getState()));
            currentState.exitFromState();
            currentState = newState;
            currentState.enterToState();
        }
    }

    protected void turnToState(int newQInformGridState) throws IncorrectGridActionPerformed {
        QInformGridState state;
        switch(newQInformGridState) {
            case QInformGridState.EDIT_STATE: state=editState; break;
            case QInformGridState.SEARCH_STATE: state=searchState; break;
            case QInformGridState.SELECTED_STATE: state=selectedState; break;
            default: throw new IncorrectGridActionPerformed(INCORRECT_STATE_CHANGE);
        }
        turnToState(state);
    }

    public void onAction(int actionType) {
        try {
//            System.out.println("<QInformGridControllerImpl.actionPerformed.35>action = " + getActionName(action) + ", captured in :" + getStateName(currentState.getState()));
            currentState.actionPerformed(actionType);
        } catch (IncorrectGridActionPerformed incorrectGridActionPerformed) {
            DialogHelper.showModalErrorDialog(incorrectGridActionPerformed);
        }
    }

    public void onSearchRequest(RecordFilter filter) {
        sendRequest(getSearchRequest(filter));
    }

    public void onMoreDataRequest(FieldDataRequest dataRequest) {
        sendRequest(getControlDataRequest(dataRequest));
    }

    public void addRecordsToModel(List rows, Collection filters) {
        for(int i = 0; i < rows.size(); i++) {//todo implement insertRows(List)
            RowData rowData = (RowData) rows.get(i);
            model.getData().getGridData().insertRow(rowData);
            model.getData().setFilters(filters);
        }
        model.fireDataChanged();
    }

    private InFormDataRequest getSearchRequest(RecordFilter filter) {
        if(searchReq == null) {
            InFormGridFieldMeta fieldMeta = model.getMeta();
            String entityId = fieldMeta.getFilteringFormMeta().getEntityMeta()
                    .getEntityID();
            searchReq = new SearchInformRequest(fieldMeta.getFieldID(),
                    filter.getFiltersList(), entityId);
        } else {
            searchReq.setFieldsFilters(filter.getFiltersList());
        }
        return searchReq;
    }

    private InFormDataRequest getControlDataRequest(
            FieldDataRequest dataRequest) {
        if(getControlDataReq == null) {
            InFormGridFieldMeta fieldMeta = model.getMeta();
            String entityId = fieldMeta.getPresentationEntityName();
            getControlDataReq = new GetControlDataInformRequest(fieldMeta.getFieldID(),
                    dataRequest, entityId);
        } else {
            getControlDataReq.setRequest(dataRequest);
        }
        return getControlDataReq;
    }

    private InFormDataRequest getRecordsDataRequest(
            RecordFilter filter) {
        if(getRecordsDataReq == null) {
            InFormGridFieldMeta fieldMeta = model.getMeta();
            String entityId = fieldMeta.getFilteringFormMeta().getEntityMeta()
                    .getEntityID();
            getRecordsDataReq = new GetRecordsDataInformRequest(fieldMeta.getFieldID(),
                    filter, entityId);
        } else {
            getRecordsDataReq.setFilter(filter);
        }
        return getRecordsDataReq;
    }

    private void sendRequest(InFormDataRequest request) {
        dataRequestListener.needMoreData(request);
    }

    public QFormElementModel getModel() {
        return model;
    }

    public QFormElementView getView() {
        return view;
    }

    private class InformGridStateContext implements StateContext {
        public void changeState(int informGridState) throws IncorrectGridActionPerformed {
            turnToState(informGridState);
        }

        public void enableLinkButtons() {
            view.setButtonsEnabled(QInformGrid.LINK_BUTTON_INDEX, true);
            view.setButtonsEnabled(QInformGrid.UNLINK_BUTTON_INDEX, true);
        }

        public void disableLinkButtons() {
            view.setButtonsEnabled(QInformGrid.LINK_BUTTON_INDEX, false);
            view.setButtonsEnabled(QInformGrid.UNLINK_BUTTON_INDEX, false);
        }

        public void setEnabledUnLinkButton(boolean isEnabled) {
            view.setButtonsEnabled(QInformGrid.UNLINK_BUTTON_INDEX, isEnabled);
        }

        public void showLinkButtons() {
            view.setButtonVisible(QInformGrid.LINK_BUTTON_INDEX, true);
            view.setButtonVisible(QInformGrid.UNLINK_BUTTON_INDEX, true);
        }

        public void showEnabledFilterButton() {
            view.setButtonVisible(QInformGrid.FILTERS_BUTTON_INDEX, true);
            view.setButtonsEnabled(QInformGrid.FILTERS_BUTTON_INDEX, true);
        }

        public void hideFilterButton() {
            view.setButtonVisible(QInformGrid.FILTERS_BUTTON_INDEX, false);
        }

        public void hideLinkButtons() {
            view.setButtonVisible(QInformGrid.LINK_BUTTON_INDEX, false);
            view.setButtonVisible(QInformGrid.UNLINK_BUTTON_INDEX, false);
        }

        public int getControlState() {
            return view.getViewMode();
        }

        public void showFilteringDialog() {
            view.showFilteringDialog();
        }

        public RecordFilter getDialogFilter() {
            return view.getDialogFilter();
        }

        public void setFilter(RecordFilter filter) {
            model.getData().clear();
            sendRequest(getRecordsDataRequest(filter));
        }

        public void addFilter(RecordFilter filter) {
            if(!filter.isRecordsFilter()) {
                showMessage("Couldn't add records by the filters.");
            } else {
                sendRequest(getRecordsDataRequest(filter));
            }
        }

        public void showMessage(String msg) {
            DialogHelper.showMessageDialog(msg);
        }
        /**
         * @return List<Integer> of selected indexes
         */
        public List getSelectedRecords() {
            return view.getSelectedRecords();
        }

        public void removeRecords(List records) {
            boolean wasDeleted = false;
            for(int i = 0; i < records.size(); i++) {//todo optimize it. implement in GridData delete(List rows)
                Integer integer = (Integer) records.get(i);
                wasDeleted = wasDeleted || model.getData().getGridData().removeRow(integer.intValue());
            }
            if(wasDeleted) {
                model.fireDataChanged();
            }
        }
    }

    /*private String getStateName(int state) {
        switch(state) {
//            case QInformGridState.NEW_STATE:        return "NEW_STATE:     ";
            case QInformGridState.EDIT_STATE:       return "EDIT_STATE:    ";
            case QInformGridState.SEARCH_STATE:     return "SEARCH_STATE:  ";
            case QInformGridState.SELECTED_STATE:   return "SELECTED_STATE:";
//            case QInformGridState.VIEW_STATE:       return "VIEW_STATE:    ";
        }
        return "unknown state" + state;
    }

    private String getActionName(int action) {
        switch(action) {
            *//*case QInformGrid.DELETE_BUTTON_INDEX      :return "DELETE_BUTTON_INDEX      ";
            case QInformGrid.CLEAR_BUTTON_INDEX       :return "CLEAR_BUTTON_INDEX       ";
            case QInformGrid.NEW_BUTTON_INDEX         :return "NEW_BUTTON_INDEX         ";
            case QInformGrid.CHANGE_BUTTON_INDEX      :return "CHANGE_BUTTON_INDEX      ";
            case QInformGrid.UPDATE_BUTTON_INDEX      :return "UPDATE_BUTTON_INDEX      ";
            case QInformGrid.ENABLED_FOR_EDIT_INDEX   :return "ENABLED_FOR_EDIT_INDEX   ";
            case QInformGrid.ENABLED_FOR_SEARCH_INDEX :return "ENABLED_FOR_SEARCH_INDEX ";
            case QInformGrid.DISABLED_INDEX           :return "DISABLED_INDEX           ";*//*
            case QInformGrid.RECORD_SELECTION_CHANGED_INDEX    :return "RECORD_SELECTION_CHANGED_INDEX    ";
            case QInformGrid.RECORD_UNSELECTED_INDEX  :return "RECORD_UNSELECTED_INDEX  ";
        }
        return "unknown action" + action;
    }
*/
}
