/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.core.client.controls.tree;

import com.google.gwt.user.client.ui.TreeListener;
import com.google.gwt.user.client.ui.TreeItem;
import com.google.gwt.core.client.GWT;
import com.queplix.core.client.app.vo.FamgMeta;
import com.queplix.core.client.app.vo.FocusMeta;
import java.util.ArrayList;

/**
 * @author: Vasily Mikhailitchenko
 * @since: 21.09.2006
 */
class QTreeControllerImpl implements QTreeController, TreeListener {
    private QTreeModelImpl model;
    private QTreeViewImpl view;
    private ArrayList nodeListeners;

    private boolean ignoreTreeItemSelectedEvent;

    public QTreeControllerImpl(QTreeModelImpl model, QTreeViewImpl view){
        this.model = model;
        this.view = view;

        this.view.getTree().addTreeListener(this);
    }
    
    /**
     * Adds a node listener to the QTree
     * @param listener - listener to add
     */
    public void addNodeListener(QTreeNodeListener listener){
        if(nodeListeners == null) {
            nodeListeners = new ArrayList();
        }
        nodeListeners.add(listener);
    }
    
    /**
     * Removes a node listener from the QTree
     * @param listener - listener to remove
     */
    public void removeNodeListener(QTreeNodeListener listener){
        if(nodeListeners != null)
            nodeListeners.remove(listener);
    }

    /**
     * React on clicks on tree items
     * @param item tree item to watch
     */
    public void onTreeItemSelected(TreeItem item){
        if (ignoreTreeItemSelectedEvent) {
            return;
        }
//        GWT.log("Tree item " + item.getText() + " was selected", null);

        ArrayList selectionHistory = new ArrayList();
        
        TreeItem tmp = item, parent;
        while (tmp != null) {
            parent = tmp.getParentItem();
            if(parent != null) {
                selectionHistory.add(new Integer(parent.getChildIndex(tmp)));
            }
            tmp = parent;
        }
        
        QTreeNode selectedNode = view.getRootNode();
        
        for(int i=selectionHistory.size()-1; i>=0; i--) {
            selectedNode = selectedNode.getChild(((Integer)selectionHistory.get(i)).intValue());
        }

        for(int i = 0; i < nodeListeners.size(); i++) {
            ((QTreeNodeListener) nodeListeners.get(i)).onQTreeNodeSelected(selectedNode);
        }
    }

    /**
     * React on item opened/closed switches
     * @param item tree item to watch
     */
    public void onTreeItemStateChanged(TreeItem item){
//        GWT.log("Tree item " + item.getText() + " state changed", null);
    }

    public void selectItem(FocusMeta.Index index) {
        ignoreTreeItemSelectedEvent = true;
        view.selectItem(index);
        ignoreTreeItemSelectedEvent = false;
    }
}
