/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package com.queplix.qwoss.inbox.utils.filters;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.modules.inbox.InboxMessage;
import com.queplix.core.modules.inbox.utils.AbstractMailFilter;
import com.queplix.core.modules.inbox.utils.db.DBRealmManager;
import com.queplix.core.modules.inbox.utils.log.AbstractInboxLogPublisher;
import com.queplix.core.modules.inbox.utils.log.RoutingLogPublisher;
import com.queplix.core.modules.mail.MailAddress;
import com.queplix.core.utils.StringHelper;
import com.queplix.core.utils.JNDINames;
import com.queplix.core.utils.cache.CacheObjectManager;
import com.queplix.core.utils.sql.SqlWrapper;
import com.queplix.core.utils.sql.SqlWrapperFactory;
import com.queplix.qwoss.utils.ApplicationHelper;
import com.queplix.qwoss.inbox.ejb.InboxManagerLocal;
import com.queplix.qwoss.inbox.ejb.InboxManagerLocalHome;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * Mail filter.
 * @author Konstantin Mironov
 * @since 8 Dec 2006
 */
public class RoutingFilter extends AbstractMailFilter {

    // ------------------------------------------------------- Main method

    /*
     * No javadoc
     * @see MailFilter#filterMail
     */
    public boolean filterMail(InboxMessage inboxMessage) {

        inboxPublisher.DEBUG("Email Routing filter is started...");

        // Getting the message data.
        long accountID = inboxMessage.getAccountId().longValue();
        String to = MailAddress.toRfcString(inboxMessage.getTo());
        String from = inboxMessage.getFrom().toRfcString();
        String subject = inboxMessage.getSubject();
        String body = inboxMessage.getBody();

        // Init log data.
        String logMessage;
        String logDetailMessage = null;
        String logID = makeLogID(inboxMessage.getMessageId());

        // =========================
        DEBUG(logID + "Applying routing rules.");

        // SQL initialization.
        String sqlRules = DBRealmManager.getSql("get_routing_rules");
        SqlWrapper sqlWrapper = SqlWrapperFactory.getSqlWrapper();
        Connection con = null;
        PreparedStatement stat = null;

        // Flag - indicates end of main cycle.
        boolean stopChecking = false;

        // Go!
        try {
            // Connecting.
            con = sqlWrapper.doConnection();

            // if the object is known skip routing rules
            // if the object isn't an interaction 
            if (inboxMessage.getObjectId() == null ||
                    (inboxMessage.getObjectId() != null && inboxMessage.getObjectType() != ApplicationHelper.TICKET_OBJECT_TYPE))
            {

                // Getting filters list for this account.
                stat = sqlWrapper.doPreparedStatement(con, sqlRules);
                stat.setLong(1, accountID);
                ResultSet rs = sqlWrapper.executeQuery(stat);

                // Main cycle.
                while (rs.next() && !stopChecking) {

                    // Getting forward rule ID.
                    long ruleId = rs.getLong(1);

                    // What to check for?
                    String word = rs.getString(2);
                    if (StringHelper.isEmpty(word)) {
                        DEBUG(logID + "The rule #" + ruleId + " was skipped because of the keyword field is empty.");
                        continue;
                    } // if (StringHelper.isEmpty(word))

                    // Where to check?
                    boolean checkFrom = (rs.getInt(3) > 0);
                    boolean checkTo = (rs.getInt(4) > 0);
                    boolean checkSubject = (rs.getInt(5) > 0);
                    boolean checkBody = (rs.getInt(6) > 0);

                    // Owner ID.
                    Long ownerID = sqlWrapper.getLongParser().getValue(rs, 7);
                    // Workgroup ID.
                    Long workgroupID = sqlWrapper.getLongParser().getValue(rs, 8);
                    // Autoreply ID.
                    Long autoReplyID = sqlWrapper.getLongParser().getValue(rs, 9);

                    // Checking...
                    if (checkFrom && StringHelper.find(from, word)) {
                        logDetailMessage = "The keyword '" + word + "' was found in the 'from' field.";
                        stopChecking = true;
                    } else if (checkTo && StringHelper.find(to, word)) {
                        logDetailMessage = "The keyword '" + word + "' found in the 'to' field.";
                        stopChecking = true;
                    } else if (checkSubject && StringHelper.find(subject, word)) {
                        logDetailMessage = "The keyword '" + word + "' found in the message subject.";
                        stopChecking = true;
                    } else if (checkBody && StringHelper.find(body, word)) {
                        logDetailMessage = "The keyword '" + word + "' found in the message body.";
                        stopChecking = true;
                    } // if (checkFrom && StringHelper.find(from, word))

                    // Was the routing rule fired?
                    if (stopChecking) {
                        // Set routing info and other attributes.
                        inboxMessage.setRoutingInfo(ownerID, workgroupID);
                        inboxMessage.setRoutingRuleId(new Long(ruleId));
                        inboxMessage.setAutoreplyId(autoReplyID);

                        // Logging.
                        logMessage = "Message was routed. ";
                        INFO(logID + logMessage + logDetailMessage + ". The rule #" + ruleId + ". Message: " + inboxMessage);
                        INFO(logID + ". Set owner #" + ownerID + " and workgroup #" + workgroupID);
                        inboxPublisher.INFO(logMessage + logDetailMessage, inboxMessage);
                        break;
                    } // if (stopChecking)
                } // while (rs.next() && !stopChecking)

                rs.close();
                stat.close();
            } else {
                // the object is already known
                stopChecking = true;
            } // if (inboxMessage.getObjectId() == null)

            if (stopChecking) {
                if (inboxMessage.getResipient() == null && inboxMessage.getWorkgroupId() == null) {
                    getDefaultRoutingInfo(inboxMessage, stat, con, logID);
                }
                InboxManagerLocal local = getLocalInboxManager();
                // is a tiket has already exist
                if (inboxMessage.getObjectId() == null) {
                    // create new ticket
                    local.createTicket(getLogonSession(), inboxMessage);
                } else {
                    if (inboxMessage.getObjectId() != null && inboxMessage.getObjectType() == ApplicationHelper.TICKET_OBJECT_TYPE) {
                        local.updateTicket(getLogonSession(), inboxMessage);
                    } else {
                        local.createTicket(getLogonSession(), inboxMessage);
                    } // if (message.getObjectId() != null && message.getObjectType() == ApplicationHelper.TICKET_OBJECT_TYPE)
                } // if (message.getObjectId() == null)
                // link attachments to the ticket
                if (inboxMessage.getObjectId() != null && inboxMessage.getProcessId() != null) {
                    local.linkAttachmentsToObject(getLogonSession(), inboxMessage);
                }
            } // if (stopChecking)

            // next filter should be the Apply filter save the Message into the My Queue
            // set internal type of message
            inboxMessage.setMessageObjectType(ApplicationHelper.MESSAGE_OBJECT_TYPE);

        } catch(SQLException sqlex) {
            // Logging and re-throwing the exception.
            logMessage = "Routing failed due to SQL error: " + sqlex.getMessage();
            inboxPublisher.ERROR(logMessage, inboxMessage);
            ERROR(logID + logMessage);

            throw new GenericSystemException(sqlex);
        } finally {
            // Close connection.
            sqlWrapper.closeConnection(con, stat);

        } // try-catch

        // Ok.
        inboxPublisher.DEBUG("Email Routing filter was finished...");
        return true;

    } // filterMail()

    // ------------------------------------------------------- Protected methods

    protected boolean getDefaultRoutingInfo(InboxMessage message, PreparedStatement stat, Connection con, String logID) {

        String sqlDefInfo = DBRealmManager.getSql("get_default_routing_info");
        SqlWrapper sqlWrapper = SqlWrapperFactory.getSqlWrapper();

        DEBUG( logID + "Checking for default routing info." );

        try {
            stat = sqlWrapper.doPreparedStatement(con, sqlDefInfo);
            stat.setLong(1, message.getAccountId().longValue());
            ResultSet rs = sqlWrapper.executeQuery(stat);
            if (!rs.next()) {
                throw new NullPointerException("Cannot load default routing " +
                    "info for account #" + message.getAccountId().longValue());
            } // if (!rs.next())

            // Default Owner ID.
            Long ownerID = sqlWrapper.getLongParser().getValue(rs, 1);
            // Default Workgroup ID.
            Long workgroupID = sqlWrapper.getLongParser().getValue(rs, 2);
            // Set routing info.
            message.setRoutingInfo(ownerID, workgroupID);

            // Logging.
            String logMessage = "The message was routed to the default owner (from account with ID = " + message.getAccountId().longValue() + ").";
            INFO(logID + logMessage + ". Message: " + message);
            INFO(logID + ". Set owner #" + ownerID + " and workgroup #" + workgroupID);
            inboxPublisher.INFO(logMessage, message);
        } catch (SQLException sqlex) {
            
            // Logging and re-throwing the exception.
            String logMessage = "Routing failed due to SQL error: " + sqlex.getMessage();
            inboxPublisher.ERROR(logMessage, message);
            ERROR(logID + logMessage);
            throw new GenericSystemException(sqlex);
        } // try
        return true;
    } // getDefaultRoutingInfo(InboxMessage) : boolean
    
    /*
     * No javadoc
     * @see AbstractMailFilter#createInboxLogPublisher
     */
    protected AbstractInboxLogPublisher createInboxLogPublisher() {
        return new RoutingLogPublisher( getLogonSession() );
    }

    /**
     * InboxManagerLocal getter.
     * @return InboxManagerLocal
     */
    protected InboxManagerLocal getLocalInboxManager() {
        CacheObjectManager com = new CacheObjectManager();
        return (InboxManagerLocal)com.getLocalObject(JNDINames.InboxManager, InboxManagerLocalHome.class);
    } // getInboxManager() : InboxManagerLocal

} // end of class
