/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */

package com.queplix.qwoss.notification;

import java.util.List;

import com.queplix.core.error.ErrorHelper;
import com.queplix.core.modules.eql.error.EQLException;
import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.integrator.security.User;
import com.queplix.core.integrator.security.WorkGroup;
import com.queplix.qwoss.gen.NotifPriorityFieldObject;
import com.queplix.qwoss.gen.NotifPriorityFieldObjectHandler;
import com.queplix.qwoss.gen.NotifRecipientFieldObject;
import com.queplix.qwoss.gen.NotifRecipientFieldObjectHandler;
import com.queplix.qwoss.gen.NotifStatusFieldObject;
import com.queplix.qwoss.gen.NotifStatusFieldObjectHandler;
import com.queplix.qwoss.gen.TicketNotificationObject;
import com.queplix.qwoss.gen.TicketNotificationObjectHandler;
import com.queplix.qwoss.gen.TicketObject;
import com.queplix.qwoss.gen.TicketObjectHandler;
import com.queplix.qwoss.utils.TicketNotificationHelper;
import com.queplix.qwoss.utils.ApplicationHelper;

/**
 * <p>
 * Ticket notification manager EJB
 * </p>
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.8.2.2 $ $Date: 2005/12/20 16:43:05 $
 */

public class TicketNotificationManagerEJB
    extends
    AbstractEntityNotificationManagerEJB {

    // ----------------------------------------------------- variables

    // base attributes
    private TicketObject ticketObject;

    // ...special
    private User agent;
    private User owner;
    private User intCustomer;
    private String customerMail;
    private WorkGroup group;

    //private Form form;

    // ----------------------------------------------------- create method

    /**
     * Initialize bean
     *
     * @param ls LogonSession object
     * @param caseObject case JEO
     */
    public void ejbCreate( LogonSession ls, TicketObject ticketObject ) {
        INFO( "TicketNotificationManagerEJB create - " + hashCode() );

        // checking
        if( ticketObject == null ) {
            throw new IllegalStateException( "Ticket object is NULL" );
        }

        // initialization
        init( ls, TicketObjectHandler.ENTITY, ticketObject.getQw_ticketid(), ApplicationHelper.TICKET_OBJECT_TYPE );
        this.ticketObject = ticketObject;
    }

    /**
     * Initialize bean
     *
     * @param ls  LogonSession object
     * @param ticketId  ticket id
     */
    public void ejbCreate( LogonSession ls, Long ticketId ) {
        INFO( "TicketNotificationManagerEJB create - " + hashCode() );
        // checking
        if( ticketId == null ) {
            throw new IllegalStateException( "Ticket ID is NULL" );
        }
        // initialization
        init( ls, TicketObjectHandler.ENTITY, ticketId, ApplicationHelper.TICKET_OBJECT_TYPE );
        TicketObjectHandler hnd = null;
        try {
            hnd = ( TicketObjectHandler ) TicketObjectHandler.selectByID(
                jeoManager, ls, ticketId.longValue() );
        } catch( EQLException ex ) {
            throwException( ex );
        }
        if( hnd == null ) {
            throwException( "Case #" + ticketId + " not found!" );
        }
        this.ticketObject = ( TicketObject ) hnd.getJEObject();
    }

    // ----------------------------------------------------- busines methods

    /**
     * Send ticket notification
     */
    public void doNotification() {

        long time = System.currentTimeMillis();
        boolean isIntCustomer = false;
        Long ticketCustomerId =  ticketObject.getQw_customerid();
        if(ticketCustomerId == null){
            ticketCustomerId = ticketObject.getQw_employeeid();
            isIntCustomer = true;
        }    
        
        if( getLogger().isInfoEnabled() ) {
            INFO( "Try to do case notification:" );
            INFO( "   Pkey: " + ticketObject.getQw_ticketid() );
            INFO( "   Priority: " + ticketObject.getQw_priority() );
            INFO( "   Status: " + ticketObject.getQw_status() );
            INFO( "   Product: " + ticketObject.getQw_productid() );
            INFO( "   Ticket Customer: " + ticketCustomerId );
            INFO( "   Owner: " + ticketObject.getQw_ownerid() );
            INFO( "   Agent: " + ticketObject.getQw_agentid() );
        }

        // Initialization
        NotificationManagerLocal notificationLocal = getNotificationManagerLocal();

        // Go!
        try {
            //
            // 1. Find ticket notifications by the ticket parameters
            //
            List notificationObjectList = TicketNotificationObjectHandler
                .selectByTicketData( jeoManager, ls, ticketObject.getQw_productid(),
                        ticketCustomerId, isIntCustomer );
            if( notificationObjectList != null ) {
                //
                // 2. For each notification do the following...
                //
                int size = notificationObjectList.size();
                for( int i = 0; i < size; i++ ) {
                    TicketNotificationObjectHandler hnd = ( TicketNotificationObjectHandler )
                        notificationObjectList.get( i );
                    TicketNotificationObject notification = ( TicketNotificationObject )
                        hnd.getJEObject();
                    // Do not send notification, if disabled
                    if( notification.getQw_disable().intValue() == 1 ) {
                        INFO( "Skip notification process. Ticket action is disabled." );
                        continue;
                    }
                    Long notificationID = notification.getQw_notificationid();

                    Integer notificationStatus = TicketNotificationHelper
                        .ticketStatus2NotificationStatus( ticketObject.getQw_status() );

                    // Check for equal status
                    List notificationStatusHandlers = NotifStatusFieldObjectHandler
                        .selectByNotificationId( jeoManager, ls, notificationID );

                    if( notificationStatusHandlers != null ) {
                        boolean returned = false;
                        for( int j = 0; j < notificationStatusHandlers.size(); j++ ) {
                            NotifStatusFieldObjectHandler notifStatusFieldObjectHandler = ( NotifStatusFieldObjectHandler ) 
                                notificationStatusHandlers.get( j );
                            NotifStatusFieldObject notifStatusFieldObject = ( NotifStatusFieldObject ) 
                                notifStatusFieldObjectHandler.getJEObject();
                            if( ( notificationStatus != null ) &&
                                ( notifStatusFieldObject.getQw_status().intValue() == notificationStatus.intValue() ) ) {
                                returned = true;
                                break;
                            }
                        }
                        if( !returned ) {
                            continue;
                        }
                    }

                    Integer notificationPriority = TicketNotificationHelper
                        .ticketPriority2NotificationPriority( ticketObject.getQw_priority() );

                    // Check for equal priority
                    List notificationPriorityHandlers = NotifPriorityFieldObjectHandler
                        .selectByNotificationId( jeoManager, ls, notificationID );

                    if( notificationPriorityHandlers != null ) {
                        boolean returned = false;
                        for( int j = 0; j < notificationPriorityHandlers.size(); j++ ) {
                            NotifPriorityFieldObjectHandler notifPriorityFieldObjectHandler =
                                ( NotifPriorityFieldObjectHandler ) notificationPriorityHandlers.get( j );
                            NotifPriorityFieldObject notifPriorityFieldObject =
                                ( NotifPriorityFieldObject ) notifPriorityFieldObjectHandler.getJEObject();
                            if( ( notificationPriority != null ) &&
                                ( notifPriorityFieldObject.getQw_priority().intValue() == notificationPriority.intValue() ) ) {
                                returned = true;
                                break;
                            }
                        }
                        if( !returned ) {
                            continue;
                        }
                    }

                    //
                    // 3. Get notification recipients
                    //
                    // get Notification ID
                    // selects the Notification
                    List notifRecipientHandlers = NotifRecipientFieldObjectHandler
                        .selectByNotificationId( jeoManager, ls, notificationID );

                    //
                    // 4. Set optional notification attributes
                    //
                    notificationLocal.setAlertSeverityId(
                            TicketNotificationHelper.ticketPriority2AlertSeverity(ticketObject.getQw_priority()));

                    notificationLocal.setSubject( notification.getQw_subject() );
                    //INFO("RUTEST: "+notification.getQw_mailbody());
                    //notificationLocal.setMessage("TEST MSG MMMMMM ");
                    notificationLocal.setMessage( ( notification.getQw_mailbody() == null ) ?
                            null : new String( notification.getQw_mailbody() ) );
                    
                    //if( notification.getIncluderecord().intValue() == 1 ) {
                        // add record info
                    //    notificationLocal.setForm( loadCaseForm() );
                    //} else {
                        // reset record info
                    //    notificationLocal.setForm( null );
                    //}

                    // All recipients are in
                    if( notifRecipientHandlers != null ) {

                        // for all recipients do --
                        for( int j2 = 0; j2 < notifRecipientHandlers.size(); j2++ ) {

                            NotifRecipientFieldObjectHandler notifRecipientFieldObjectHandler = 
                                ( NotifRecipientFieldObjectHandler ) notifRecipientHandlers.get( j2 );
                            NotifRecipientFieldObject notifRecipientFieldObject = 
                                ( NotifRecipientFieldObject ) notifRecipientFieldObjectHandler.getJEObject();
                            Integer notifRecipient = notifRecipientFieldObject.getQw_recipient();

                            if( notifRecipient == null ) {
                                continue;
                            }

                            User notificationUser = null;
                            WorkGroup notificationGroup = null;
                            String notificationAddr = null;

                            // choose notifier
                            if( notifRecipient.intValue() == TicketNotificationHelper.CUSTOMER_RECIPIENT ) {
                                // to case customer..
                                if(isIntCustomer)
                                    notificationUser = loadTicketIntCustomer(ticketCustomerId);
                                else
                                    notificationAddr = loadTicketCustomer(ticketCustomerId);
                            } else if( notifRecipient.intValue() == TicketNotificationHelper.AGENT_RECIPIENT ) {
                                // to ticket agent..
                                notificationUser = loadTicketAgent();                                
                            } else if( notifRecipient.intValue() == TicketNotificationHelper.OWNER_RECIPIENT ) {
                                // to ticket owner..
                                notificationUser = loadTicketOwner();
                            } else if( notifRecipient.intValue() == TicketNotificationHelper.WORKGROUP_RECIPIENT ) {
                                // to ticket workgroup..
                                notificationGroup = loadTicketGroup();
                            } else if( notifRecipient.intValue() == TicketNotificationHelper.EMAIL_TO_RECIPIENT ) {
                                // send special email..
                                notificationAddr = notification.getQw_email();
                            } 

                            if( getLogger().isInfoEnabled() ) {
                                INFO( "Notification action #" + i );
                                INFO( "   recipient type: " + notifRecipient );
                                INFO( "   method: " + notification.getQw_sendmode() );
                                INFO( "   notification user: " + notificationUser );
                                INFO( "   notification group: " + notificationGroup );
                                INFO( "   notification addr: " + notificationAddr );
                            }

                            //
                            // 5. Get notification (action) method. Do this
                            // action.
                            //
                            if( notificationAddr != null ) {
                                //
                                // 5.1 always send email to addressee
                                //
                                notificationLocal
                                    .notifyByMail( notificationAddr );

                            } else if( notification.getQw_sendmode() == null ) {

                                //
                                // 5.2 send default notification
                                //
                                if( notificationUser != null ) {
                                    // send to user
                                    notificationLocal.notify( notificationUser );

                                } else if( notificationGroup != null ) {
                                    // send to group
                                    notificationLocal.notify( notificationGroup,
                                        null, null );

                                } else {
                                    // warning!
                                    WARN( "Can't send default notification. Recipient is empty." );
                                }

                            } else if( notification.getQw_sendmode().intValue() == TicketNotificationHelper.ALERT_METHOD ) {

                                //
                                // 5.3 send notification alert
                                //

                                if( notificationUser != null ) {
                                    // send alert to user
                                    notificationLocal
                                        .notifyByAlert( notificationUser );

                                } else if( notificationGroup != null ) {
                                    // send alert to group
                                    notificationLocal.notifyByAlert(
                                        notificationGroup, null );

                                } else {
                                    // warning!
                                    WARN( "Can't send notification alert. Recipient is empty." );
                                }

                            } else {

                                //
                                // 5.4 send notification mail
                                //

                                if( notificationUser != null ) {
                                    // send mail to user
                                    notificationLocal
                                        .notifyByMail( notificationUser );

                                } else if( notificationGroup != null ) {
                                    // send mail to group
                                    notificationLocal
                                        .notifyByMail( notificationGroup );

                                } else {
                                    // warning!
                                    WARN( "Can't send notification mail. Recipient is empty." );
                                }
                            }
                        } // --- for all recipients
                    } else {
                        WARN( "Can't send notification mail. Recipient is empty." );
                    }
                }
            }

        } catch( EQLException ex ) {
            throwException( ex );

        } catch( Throwable t ) {
            ErrorHelper.throwSystemException( t, this );

        } finally {
            // remove NotificationManager stateful EJB
            try {
                notificationLocal.remove();
            } catch( Throwable t ) {
                ERROR( t );
            }
        }

        // Ok.
        if( getLogger().isInfoEnabled() ) {
            INFO( "Ticket notification done (pkey = " + ticketObject.getQw_ticketid()                   
                  + ")." );
            INFO( "Time (ms) - " + ( System.currentTimeMillis() - time ) );
        }
    }

    /**
     * Send case owner notification
     */
    public void doOwnerNotification() {
        /** @todo implement it */
    }

    // ----------------------------------------------------- private methods

    //
    // Load ticket agent
    //
    private User loadTicketAgent() {
        if( agent == null ) {
            INFO( "Trying to load ticket agent #" + ticketObject.getQw_agentid() );
            agent = loadUser(  ticketObject.getQw_agentid() );
        }
        return agent;
    }
    
    //
    // Load ticket owner
    //
    private User loadTicketOwner() {
        if( owner == null ) {
            INFO( "Trying to load ticket owner #" + ticketObject.getQw_ownerid() );
            owner = loadUser(  ticketObject.getQw_ownerid() );
        }
        return owner;
    }

    //
    // Load ticket Int Customer
    //
    private User loadTicketIntCustomer(Long intCustomerId) {
        if( intCustomer == null ) {
            INFO( "Trying to load ticket internal customer #" + intCustomerId );
            intCustomer = loadEmployee( intCustomerId );
        }
        return intCustomer;
    }

    //
    // Load ticket Customer
    //
    private String loadTicketCustomer(Long customerId) {
        if( customerMail == null ) {
            INFO( "Trying to load mail address for ticket internal customer #" + customerId );
            customerMail = loadCustomerMail( customerId );
            //customerMail = "test@mail.ru";
        }
        return customerMail;
    }
    
    //
    // Load ticket group
    //
    private WorkGroup loadTicketGroup() {
        if( group == null ) {
            INFO( "Trying to load ticket group #" + ticketObject.getQw_workgroupid() );
            group = loadGroup( ticketObject.getQw_workgroupid() );
        }
        return group;
    }

}
