/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.config.ejb;

import com.queplix.core.modules.config.jxb.Custom;
import com.queplix.core.modules.config.jxb.CustomField;
import com.queplix.core.modules.config.utils.ConfigPropertyFactory;
import com.queplix.core.modules.config.utils.CustomID;
import com.queplix.core.utils.cache.Cache;
import com.queplix.core.utils.ejb.AbstractSessionEJB;

/**
 * Facade session EJB for Custom management
 *
 * @author [ALB] Baranov Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:14 $
 */

public class CustomConfigManagerEJB
        extends AbstractSessionEJB {

    //---------------------------------------------------------------- public methods

    /**
     * Initialize bean.
     */
    public void ejbCreate() {
    }

    /**
     * Fill in the collection of Custom objects.
     *
     * @param customs array of Custom objects
     * @return number of updated records
     */
    public int fillCustomConfigs(Custom[] customs) {

        long time = System.currentTimeMillis();
        DEBUG("Try to fill all custom configs");

        // Clear cache.
        Cache cache = ConfigPropertyFactory.getInstance()
                .getCustomConfigCache();
        cache.clear();

        // Fill Customs object.
        int updated = ConfigPropertyFactory.getInstance().getCustomConfigDAO()
                .storeCustomsVO(customs);

        if(getLogger().isDebugEnabled()) {
            DEBUG("Fill all custom configs. Count " + updated +
                    ". Time (ms): " + (System.currentTimeMillis() - time));
        }

        return updated;
    }

    /**
     * Get Custom object with localized caption by name.
     *
     * @param langID     Language ID
     * @param entityName entity name (custom name)
     * @return Custom object
     */
    public Custom getLocalizedCustomConfig(String langID, String entityName) {

        // Try to find out in cache.
        Cache cache = ConfigPropertyFactory.getInstance()
                .getCustomConfigCache();
        CustomID key = new CustomID(entityName, langID);
        Custom custom = (Custom) cache.get(key);

        if(getLogger().isDebugEnabled()) {
            DEBUG("Try to get custom config '" + key + "'");
        }

        if(custom == null) {
            // Load record from database.
            custom = ConfigPropertyFactory.getInstance().getCustomConfigDAO()
                    .loadCustomVO(key);

            // Store in cache.
            cache.put(key, custom);

        } else {
            // Found in cache!
            if(getLogger().isDebugEnabled()) {
                DEBUG("Custom '" + key + "' found in the cache.");
            }
        }

        // Return read-only custom.
        return custom;
    }

    /**
     * Get CustomField value object with localized caption by the entity name
     * <code>name</code> and efield name <code>efieldName</code>
     *
     * @param langID     Language ID
     * @param entityName entity name (custom name)
     * @param efieldName efield name
     * @return Custom object
     */
    public CustomField getLocalizedCustomField(String langID,
                                               String entityName,
                                               String efieldName) {

        Custom custom = getLocalizedCustomConfig(langID, entityName);
        if(custom == null) {
            return null;
        }

        return (CustomField) custom.getObject(efieldName);
    }

}
