/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.config.utils.db;

import com.queplix.core.error.GenericSystemException;
import com.queplix.core.modules.config.jxb.ContextMenu;
import com.queplix.core.modules.config.jxb.ContextMenuForm;
import com.queplix.core.modules.config.jxb.MenuItem;
import com.queplix.core.modules.config.utils.ContextMenuConfigDAO;
import com.queplix.core.utils.dao.AbstractDAO;
import com.queplix.core.utils.sql.SqlWrapper;
import com.queplix.core.utils.sql.SqlWrapperFactory;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;

/**
 * Implementation of the CaptionConfigDAO
 *
 * @author [MVT] Michael Trofimov
 * @version $Revision$ $Date$
 */
public class ContextMenuConfigDAOImpl extends AbstractDAO implements
        ContextMenuConfigDAO {

    // --------------------------------------------------------------- Variables

    protected SqlWrapper sqlWrapper = SqlWrapperFactory.getSqlWrapper();

    // --------------------------------------------------------------- Overrided Methods

    public int clearContextMenusVO() {
        Connection con = null;
        PreparedStatement ps = null;
        try {
            con = sqlWrapper.doConnection();
            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "delall_menuItem"));
            sqlWrapper.executeUpdate(ps);

            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "delall_menuForm"));
            sqlWrapper.executeUpdate(ps);

            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "delall_contextMenu"));
            return sqlWrapper.executeUpdate(ps);

        } catch (SQLException ex) {
            throw new GenericSystemException(
                    "SQL exception: " + ex.getMessage(), ex);
        } finally {
            sqlWrapper.closeConnection(con, ps);
        }
    }

    public ContextMenu[] loadContextMenusVO() {
        Connection con = null;
        PreparedStatement ps = null;

        List<ContextMenu> contextMenus = new ArrayList<ContextMenu>();

        try {
            con = sqlWrapper.doConnection();
            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "select_contextMenu"));
            ResultSet rs = sqlWrapper.executeQuery(ps);
            while(rs.next()) {
                ContextMenu contextMenu = new ContextMenu();
                contextMenu.setName(sqlWrapper.getStringParser().getValue(rs,
                        1));

                contextMenus.add(contextMenu);

                loadMenuItems(con, contextMenu);
                loadMenuForms(con, contextMenu);
            }
            rs.close();
            ps.close();

        } catch (SQLException e) {
            throw new GenericSystemException("SQL exception: " + e.getMessage(),
                    e);
        } finally {
            sqlWrapper.closeConnection(con, ps);
        }

        return contextMenus.toArray(new ContextMenu[contextMenus.size()]);

    }

    public int storeContextMenusVO(ContextMenu[] contextMenus) {
        Connection con = null;
        PreparedStatement ps = null;

        int updated = 0;
        try {
            con = sqlWrapper.doConnection();

            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "insert_contextMenu"));
            for(int i = 0; i < contextMenus.length; i++) {
                ContextMenu contextMenu = contextMenus[i];

                sqlWrapper.getStringParser().setValue(ps, 1,
                        contextMenu.getName());
                updated += sqlWrapper.executeUpdate(ps);

            }
            ps.close();

            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "insert_menuItem"));
            for(ContextMenu contextMenu : contextMenus) {
                for(MenuItem menuItem : contextMenu.getMenuItem()) {
                    sqlWrapper.getStringParser().setValue(ps, 1,
                            menuItem.getName());
                    sqlWrapper.getStringParser().setValue(ps, 2,
                            menuItem.getContextMenu());
                    sqlWrapper.getIntParser().setValue(ps, 3,
                            menuItem.getOrder());
                    sqlWrapper.executeUpdate(ps);
                }
            }
            ps.close();

            ps = sqlWrapper.doPreparedStatement(con, DBRealmManager.getSql(
                    "insert_menuForm"));
            for(ContextMenu contextMenu : contextMenus) {
                for(ContextMenuForm menuForm : contextMenu
                        .getContextMenuForm()) {
                    sqlWrapper.getStringParser().setValue(ps, 1,
                            menuForm.getName());
                    sqlWrapper.getStringParser().setValue(ps, 2,
                            menuForm.getContextMenu());
                    sqlWrapper.executeUpdate(ps);
                }
            }
            ps.close();

            return updated;

        } catch (SQLException e) {
            throw new GenericSystemException("SQL exception: " + e.getMessage(),
                    e);
        } finally {
            sqlWrapper.closeConnection(con, ps);
        }
    }

    // --------------------------------------------------------------- Non-public Methods

    private void loadMenuItems(Connection con, ContextMenu contextMenu)
            throws SQLException {
        PreparedStatement ps = sqlWrapper.doPreparedStatement(con,
                DBRealmManager.getSql("select_menuItem"));

        String contextMenuName = contextMenu.getName();
        ps.setString(1, contextMenuName);
        ResultSet rs = sqlWrapper.executeQuery(ps);
        while(rs.next()) {
            MenuItem menuItem = new MenuItem();
            menuItem.setName(sqlWrapper.getStringParser().getValue(rs, 1));
            menuItem.setContextMenu(sqlWrapper.getStringParser().getValue(rs,
                    2));
            menuItem.setOrder(sqlWrapper.getIntParser().getValue(rs, 3));

            contextMenu.addMenuItem(menuItem);
//			contextMenu.putObject(menuItem.getName(), menuItem);
        }
        rs.close();
        ps.close();
    }

    private void loadMenuForms(Connection con, ContextMenu contextMenu)
            throws SQLException {
        PreparedStatement ps = sqlWrapper.doPreparedStatement(con,
                DBRealmManager.getSql("select_menuForm"));

        String contextMenuName = contextMenu.getName();
        ps.setString(1, contextMenuName);
        ResultSet rs = sqlWrapper.executeQuery(ps);
        while(rs.next()) {
            ContextMenuForm menuForm = new ContextMenuForm();
            menuForm.setName(sqlWrapper.getStringParser().getValue(rs, 1));
            menuForm.setContextMenu(sqlWrapper.getStringParser().getValue(rs,
                    2));

            contextMenu.addContextMenuForm(menuForm);
//			contextMenu.putObject(menuForm.getName(), menuForm);
        }
        rs.close();
        ps.close();
    }

}
