/*
 * Copyright 2006-2007 Queplix Corp.
 *
 * Licensed under the Queplix Public License, Version 1.1.1 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.queplix.com/solutions/commercial-open-source/queplix-public-license/
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 *
 */

package com.queplix.core.modules.eqlext.transforms.impl;

import com.queplix.core.integrator.security.LogonSession;
import com.queplix.core.integrator.security.User;
import com.queplix.core.jxb.entity.Efield;
import com.queplix.core.jxb.entity.Entity;
import com.queplix.core.jxb.entity.types.SqlSType;
import com.queplix.core.modules.eqlext.error.EfieldTransformException;
import com.queplix.core.modules.eqlext.transforms.EfieldTransform;
import com.queplix.core.utils.DateHelper;
import com.queplix.core.utils.SystemHelper;

import java.sql.Time;
import java.text.ParseException;
import java.util.TimeZone;

/**
 * <p>Default Time transform class</p>
 *
 * @author [ALB] Baranov L. Andrey
 * @version $Revision: 1.1.1.1 $ $Date: 2005/09/12 15:30:40 $
 */

public class TimeTransform
        extends EfieldTransform {

    /*
     * No javadoc
     * @see EfieldTransform#toObject
     */
    public Object toObject(LogonSession ls, Entity entity, Efield field,
                           String s)
            throws EfieldTransformException {

        // convert string to secs
        int secs;
        try {
            secs = DateHelper.parseTime(s, false);
        } catch (ParseException ex) {
            throw new EfieldTransformException(ex.getMessage(), ex);
        }

        // build POJO
        int sqlType = field.getSqltype().getType();
        switch(sqlType) {
            case SqlSType.TIME_TYPE: {
                return toSystemTime(secs);
            }
            case SqlSType.INT_TYPE: {
                return secs;
            }
            case SqlSType.LONG_TYPE: {
                return (long) secs;
            }
            default:
                throw new IllegalStateException(
                        "Unsupported sql type '" + sqlType +
                                "' for time field");
        }
    }

    /*
     * No javadoc
     * @see EfieldTransform#toString
     */
    public String toString(LogonSession ls, Entity entity, Efield field,
                           Object o)
            throws EfieldTransformException {

        User user = ls.getUser();
        String timePattern = user.getTimePattern();

        // analyse POJO
        int sqlType = field.getSqltype().getType();
        Time time;
        switch(sqlType) {
            case SqlSType.TIME_TYPE:
                time = (Time) o;
                break;

            case SqlSType.INT_TYPE:
            case SqlSType.LONG_TYPE:
                int secs = ((Number) o).intValue();
                time = toSystemTime(secs);
                break;

            default:
                throw new IllegalStateException(
                        "Unsupported sql type '" + sqlType +
                                "' for time field");
        }

        return DateHelper.formatTime(time, timePattern);
    }

    //
    // Convert default (JVM) seconds to UPC (system) time
    //
    private Time toSystemTime(int secs) {
        long systemSecs = DateHelper.convert(secs * 1000,
                TimeZone.getDefault(),
                SystemHelper.SYSTEM_TIMEZONE);
        return new Time(systemSecs);
    }
}
